package org.molgenis.controller;

import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.filter.StudyDataRequest;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.StudyDataRequestService;
import org.molgenis.omx.observ.ObservableFeature;
import org.molgenis.controller.ObservableFeatureController.ObservableFeatureResponse;
import org.molgenis.service.ObservableFeatureService;
import org.molgenis.util.EntityPager;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/studydatarequest")
public class StudyDataRequestController
{
	@Autowired
	private StudyDataRequestService studyDataRequestService;

	@Autowired
	private ObservableFeatureService observableFeatureService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<StudyDataRequestResponse> createStudyDataRequest(@Valid @RequestBody StudyDataRequestRequest studyDataRequestRequest)
			throws DatabaseException
	{
		return _createStudyDataRequest(studyDataRequestRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<StudyDataRequestResponse> createStudyDataRequestFromForm(@Valid @ModelAttribute StudyDataRequestRequest studyDataRequestRequest)
			throws DatabaseException
	{
		return _createStudyDataRequest(studyDataRequestRequest);
	}

	private ResponseEntity<StudyDataRequestResponse> _createStudyDataRequest(StudyDataRequestRequest studyDataRequestRequest) throws DatabaseException
	{
		StudyDataRequest studyDataRequest = studyDataRequestService.create(studyDataRequestRequest.toStudyDataRequest());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/studydatarequest/" + studyDataRequest.getId());
		return new ResponseEntity<StudyDataRequestResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public StudyDataRequestResponse retrieveStudyDataRequest(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequest(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public StudyDataRequestResponse retrieveStudyDataRequestJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequest(id, expandFields);
	}

	private StudyDataRequestResponse _retrieveStudyDataRequest(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		StudyDataRequest studyDataRequest = studyDataRequestService.read(id);
		if (studyDataRequest == null) throw new EntityNotFoundException("StudyDataRequest " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new StudyDataRequestResponse(studyDataRequest, expandFields);
	}
			
	@RequestMapping(value = "/{id}/molgenisUser", method = RequestMethod.GET)
	public String retrieveStudyDataRequestXrefMolgenisUser(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequestXrefMolgenisUser(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/molgenisUser", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveStudyDataRequestXrefMolgenisUserJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequestXrefMolgenisUser(id, "json", expandFields);
	}
	
	private String _retrieveStudyDataRequestXrefMolgenisUser(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		StudyDataRequest studyDataRequest = studyDataRequestService.read(id);
		if (studyDataRequest == null) throw new EntityNotFoundException("StudyDataRequest " + id.toString() + " not found");
		Integer molgenisUserId = studyDataRequest.getMolgenisUser_Id();
		String redirectUri = "redirect:/api/v1/molgenisuser/" + molgenisUserId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/features", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveStudyDataRequestMrefFeatures(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		StudyDataRequest studyDataRequest = studyDataRequestService.read(id);
		if (studyDataRequest == null) throw new EntityNotFoundException("StudyDataRequest " + id.toString() + " not found");
		return _retrieveStudyDataRequestMrefFeatures(studyDataRequest, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/features", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveStudyDataRequestMrefFeaturesJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		StudyDataRequest studyDataRequest = studyDataRequestService.read(id);
		if (studyDataRequest == null) throw new EntityNotFoundException("StudyDataRequest " + id.toString() + " not found");
		return _retrieveStudyDataRequestMrefFeatures(studyDataRequest, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<ObservableFeatureResponse> _retrieveStudyDataRequestMrefFeatures(StudyDataRequest studyDataRequest, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<ObservableFeature> observableFeatureCollection = studyDataRequest.getFeatures();
		
		int total = observableFeatureCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		observableFeatureCollection = observableFeatureCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<ObservableFeature> observableFeaturePager = new EntityPager<ObservableFeature>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, observableFeatureCollection);
		
		return new EntityCollectionResponse<ObservableFeatureResponse>(observableFeaturePager, Lists.newArrayList(Iterables.transform(observableFeatureCollection,
				new Function<ObservableFeature, ObservableFeatureResponse>()
				{
					@Override
					@Nullable
					public ObservableFeatureResponse apply(@Nullable ObservableFeature observableFeature)
					{
						try
						{
							return observableFeature != null ? new ObservableFeatureResponse(observableFeature, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/studydatarequest/" + studyDataRequest.getId() + "/features");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateStudyDataRequest(@PathVariable Integer id, @Valid @RequestBody StudyDataRequestRequest studyDataRequestRequest)
			throws DatabaseException
	{
		_updateStudyDataRequest(id, studyDataRequestRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<StudyDataRequestResponse> updateStudyDataRequestFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute StudyDataRequestRequest studyDataRequestRequest) throws DatabaseException
	{
		return _createStudyDataRequest(studyDataRequestRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateStudyDataRequestPost(@PathVariable Integer id, @Valid @RequestBody StudyDataRequestRequest studyDataRequestRequest)
			throws DatabaseException
	{
		_updateStudyDataRequest(id, studyDataRequestRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateStudyDataRequestFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute StudyDataRequestRequest studyDataRequestRequest)
			throws DatabaseException
	{
		_updateStudyDataRequest(id, studyDataRequestRequest);
	}

	private void _updateStudyDataRequest(Integer id, StudyDataRequestRequest studyDataRequestRequest) throws DatabaseException
	{
		StudyDataRequest studyDataRequest = studyDataRequestRequest.toStudyDataRequest();
		studyDataRequest.setId(id);
		studyDataRequestService.update(studyDataRequest);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteStudyDataRequest(@PathVariable Integer id) throws DatabaseException
	{
		_deleteStudyDataRequest(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteStudyDataRequestPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteStudyDataRequest(id);
	}

	private void _deleteStudyDataRequest(Integer id) throws DatabaseException
	{
		boolean isDeleted = studyDataRequestService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("StudyDataRequest " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<StudyDataRequestResponse> retrieveStudyDataRequestCollection(@Valid EntityCollectionRequest studyDataRequestCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequestCollection(studyDataRequestCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<StudyDataRequestResponse> retrieveStudyDataRequestCollectionJson(@Valid EntityCollectionRequest studyDataRequestCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequestCollection(studyDataRequestCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<StudyDataRequestResponse> retrieveStudyDataRequestCollectionPost(@Valid @RequestBody EntityCollectionRequest studyDataRequestCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequestCollection(studyDataRequestCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<StudyDataRequestResponse> retrieveStudyDataRequestCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest studyDataRequestCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStudyDataRequestCollection(studyDataRequestCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<StudyDataRequestResponse> _retrieveStudyDataRequestCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<StudyDataRequest> studyDataRequestPager = studyDataRequestService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<StudyDataRequestResponse>(studyDataRequestPager, Lists.newArrayList(Iterables.transform(
				studyDataRequestPager.getIterable(), new Function<StudyDataRequest, StudyDataRequestResponse>()
				{
					@Override
					@Nullable
					public StudyDataRequestResponse apply(@Nullable StudyDataRequest studyDataRequest)
					{
						try
						{
							return studyDataRequest != null ? new StudyDataRequestResponse(studyDataRequest, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/studydatarequest");
	}

	private static class StudyDataRequestRequest
	{
		private String identifier;
		private String name;
		private String requestForm;
		private java.util.List<Integer> features;
		private java.util.Date requestDate;
		private String requestStatus;
	
		public StudyDataRequest toStudyDataRequest()
		{
			StudyDataRequest studyDataRequest = new StudyDataRequest();
			studyDataRequest.setIdentifier(identifier);
			studyDataRequest.setName(name);
			studyDataRequest.setRequestForm(requestForm);
			studyDataRequest.setFeatures_Id(features);
			studyDataRequest.setRequestDate(requestDate);
			studyDataRequest.setRequestStatus(requestStatus);
			return studyDataRequest;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setRequestForm(String requestForm)
		{
			this.requestForm = requestForm;
		}
		
		public void setFeatures(java.util.List<Integer> features)
		{
			this.features = features;
		}
		
		
		public void setRequestDate(java.util.Date requestDate)
		{
			this.requestDate = requestDate;
		}
		
		public void setRequestStatus(String requestStatus)
		{
			this.requestStatus = requestStatus;
		}
		
	}

	static class StudyDataRequestResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String requestForm;
		private final Object features;
		private final java.util.Date requestDate;
		private final String requestStatus;
	
		public StudyDataRequestResponse(StudyDataRequest studyDataRequest, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/studydatarequest/" + studyDataRequest.getId();
			this.identifier = studyDataRequest.getIdentifier();
			this.name = studyDataRequest.getName();
			this.requestForm = studyDataRequest.getRequestForm();
			
			if (expandFields != null && expandFields.contains("features")) this.features = _retrieveStudyDataRequestMrefFeatures(studyDataRequest, new EntityCollectionRequest());
			else this.features = java.util.Collections.singletonMap("href", "/api/v1/studydatarequest/" + studyDataRequest.getId() + "/features");
			this.requestDate = studyDataRequest.getRequestDate();
			this.requestStatus = studyDataRequest.getRequestStatus();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
		public String getRequestForm()
		{
			return requestForm;
		}
	
		public Object getFeatures()
		{
			return features;
		}
	
	
		public java.util.Date getRequestDate()
		{
			return requestDate;
		}
	
		public String getRequestStatus()
		{
			return requestStatus;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}