package org.molgenis;

import org.molgenis.framework.db.Database;

import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import javax.persistence.EntityManager;

import org.apache.log4j.Logger;

import org.molgenis.omx.auth.util.PasswordHasher;
import org.molgenis.omx.auth.MolgenisUser;
import org.molgenis.omx.auth.MolgenisGroup;
import org.molgenis.omx.auth.MolgenisRoleGroupLink;
import org.molgenis.omx.auth.MolgenisRole;
import org.molgenis.omx.auth.MolgenisPermission;
import org.molgenis.omx.core.MolgenisEntity;

import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.security.Login;
import org.molgenis.framework.server.MolgenisPermissionService;
import org.molgenis.framework.ui.MolgenisPlugin;
import org.molgenis.framework.ui.MolgenisPluginRegistry;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

public abstract class MolgenisDatabasePopulator implements ApplicationListener<ContextRefreshedEvent> {
	private static final Logger logger = Logger.getLogger(MolgenisDatabasePopulator.class);
	
	// FIXME close database
	@Autowired
	@Qualifier("unauthorizedPrototypeDatabase")
	private Database database;
	
	@Value("${admin.password:@null}")
	private String adminPassword;
	@Value("${admin.email:molgenis+admin@gmail.com}")
	private String adminEmail;
	
	@Override
	public void onApplicationEvent(ContextRefreshedEvent event)
	{
		if(!isApplicationDatabaseInitialized())
		{
			try
			{
				logger.info("initializing application database defaults");
				initializeDefaultApplicationDatabase(database);
				logger.info("initialized application database defaults");
				logger.info("initializing application database");
				initializeApplicationDatabase(database);
				logger.info("initialized application database");
			}
			catch(Exception e)
			{
				throw new RuntimeException(e);
			}
		}
	}
	
	protected abstract void initializeApplicationDatabase(Database database) throws Exception;
	
	private boolean isApplicationDatabaseInitialized()
	{
		try
		{
			// the database is not initialized if it doesn't contain any users 
			return database.count(MolgenisUser.class) > 0;
		}
		catch(DatabaseException e)
		{
			throw new RuntimeException(e);
		}
	}
	
	private void initializeDefaultApplicationDatabase(Database database) throws Exception
	{
		if(adminPassword == null) throw new RuntimeException("please configure the admin.password property in your molgenis-server.properties");	
		
		Login login = database.getLogin();
    	database.setLogin(null); // so we don't run into trouble with the Security Decorators

		// add admin and anonymous users 
		MolgenisUser userAdmin = new MolgenisUser();
		userAdmin.setName(Login.USER_ADMIN_NAME);
		userAdmin.setIdentifier(UUID.randomUUID().toString());
		userAdmin.setPassword(new PasswordHasher().toMD5(adminPassword));
		userAdmin.setEmail(adminEmail);
		userAdmin.setFirstName(Login.USER_ADMIN_NAME);
		userAdmin.setLastName(Login.USER_ADMIN_NAME);
		userAdmin.setActive(true);
		userAdmin.setSuperuser(true);
		
		MolgenisUser userAnonymous = new MolgenisUser();
		userAnonymous.setName(Login.USER_ANONYMOUS_NAME);
		userAnonymous.setIdentifier(UUID.randomUUID().toString());
		userAnonymous.setPassword(new PasswordHasher().toMD5("anonymous")); 
		userAnonymous.setEmail("molgenis+anonymous@gmail.com");
		userAnonymous.setFirstName(Login.USER_ANONYMOUS_NAME);
		userAnonymous.setLastName(Login.USER_ANONYMOUS_NAME);
		userAnonymous.setActive(true);

		// add user groups for admins, normal users and the anonymous user
		MolgenisGroup systemUsersGroup = new MolgenisGroup();
		systemUsersGroup.setName(Login.GROUP_SYSTEM_NAME);
		systemUsersGroup.setIdentifier(UUID.randomUUID().toString());
				
		MolgenisGroup allUsersGroup = new MolgenisGroup();
		allUsersGroup.setName(Login.GROUP_USERS_NAME);
		allUsersGroup.setIdentifier(UUID.randomUUID().toString());

		// add links between users and groups
		MolgenisRoleGroupLink adminToSystemLink = new MolgenisRoleGroupLink();
		adminToSystemLink.setIdentifier(UUID.randomUUID().toString());
		adminToSystemLink.setName(systemUsersGroup.getName() + '-' + userAdmin.getName());
		adminToSystemLink.setGroup(systemUsersGroup);
		adminToSystemLink.setRole(userAdmin);
		
		MolgenisRoleGroupLink adminToAllUsersLink = new MolgenisRoleGroupLink();
		adminToAllUsersLink.setName(allUsersGroup.getName() + '-' + userAdmin.getName());
		adminToAllUsersLink.setIdentifier(UUID.randomUUID().toString());
		adminToAllUsersLink.setGroup(allUsersGroup);
		adminToAllUsersLink.setRole(userAdmin);
				
		// add entities to database
		EntityManager em = database.getEntityManager();
		em.getTransaction().begin();
		try
		{
	        em.persist(userAdmin);
	        em.persist(userAnonymous);
	        em.persist(systemUsersGroup);
	        em.persist(allUsersGroup);
			em.persist(adminToSystemLink);
			em.persist(adminToAllUsersLink);
	        em.getTransaction().commit();
		}
		catch(Exception e)
		{
			em.getTransaction().rollback();
			throw e;
		}
		
		login.login(database, Login.USER_ADMIN_NAME, adminPassword);

		database.beginTx();
		try
		{
			// add MolgenisEntity entities to database so we can assign permissions
			database.add(createEntities(ENTITY_VALUES));
			database.add(createEntities(UI_VALUES));
			database.add(createPluginEntities(MolgenisPluginRegistry.getInstance()));
			
			// set permissions for UI components as specified in molgenis-ui.xml
			database.commitTx();
		}
		catch(Exception e)
		{
			database.rollbackTx();
			throw e;
		}
		
		database.setLogin(login); // restore login
	}

	public MolgenisUser createUser(Database database, String userName, String firstName, String lastName, String email,
			String password, boolean superUser) throws DatabaseException
	{
		database.beginTx();
		MolgenisUser user = new MolgenisUser();
		user.setName(userName);
		user.setIdentifier(UUID.randomUUID().toString());
		user.setPassword(new PasswordHasher().toMD5(password));
		user.setEmail(email);
		user.setFirstName(firstName);
		user.setLastName(lastName);
		user.setActive(true);
		user.setSuperuser(superUser);
		database.add(user);
		database.commitTx();
		return user;
	}

	public MolgenisGroup createGroup(Database database, String groupName) throws DatabaseException
	{
		database.beginTx();
		MolgenisGroup group = new MolgenisGroup();
		group.setName(groupName);
		group.setIdentifier(UUID.randomUUID().toString());
		database.add(group);
		database.commitTx();
		return group;
		
	}

	private static List<MolgenisEntity> createPluginEntities(MolgenisPluginRegistry pluginRegistry) {
		List<MolgenisEntity> entities = new ArrayList<MolgenisEntity>();
		for(Class<? extends MolgenisPlugin> pluginClazz : pluginRegistry.getPluginClasses()) {
			MolgenisEntity entity = new MolgenisEntity();
			entity.setName(pluginClazz.getSimpleName());
			entity.setClassName(pluginClazz.getName());
			entity.setType("PLUGIN");
			entities.add(entity); 
		}
		return entities;
	}
		
	public static List<MolgenisEntity> createEntities(String[][] entityValues) {
		List<MolgenisEntity> result = new ArrayList<MolgenisEntity>(entityValues.length);
		for(String[] values : entityValues) {
			MolgenisEntity entity = new MolgenisEntity();
			entity.setName(values[0]);
			entity.setType(values[1]);
			entity.setClassName(values[2]);
			result.add(entity);      
		}		
		return result;		
	}

	private static final String[][] ENTITY_VALUES = new String[][] {
		new String[] {"MolgenisEntity", "ENTITY", "org.molgenis.omx.core.MolgenisEntity"},
		new String[] {"MolgenisFile", "ENTITY", "org.molgenis.omx.core.MolgenisFile"},
		new String[] {"RuntimeProperty", "ENTITY", "org.molgenis.omx.core.RuntimeProperty"},
		new String[] {"Characteristic", "ENTITY", "org.molgenis.omx.observ.Characteristic"},
		new String[] {"ObservationTarget", "ENTITY", "org.molgenis.omx.observ.ObservationTarget"},
		new String[] {"ObservableFeature", "ENTITY", "org.molgenis.omx.observ.ObservableFeature"},
		new String[] {"Category", "ENTITY", "org.molgenis.omx.observ.Category"},
		new String[] {"Protocol", "ENTITY", "org.molgenis.omx.observ.Protocol"},
		new String[] {"DataSet", "ENTITY", "org.molgenis.omx.observ.DataSet"},
		new String[] {"ObservationSet", "ENTITY", "org.molgenis.omx.observ.ObservationSet"},
		new String[] {"ObservedValue", "ENTITY", "org.molgenis.omx.observ.ObservedValue"},
		new String[] {"Species", "ENTITY", "org.molgenis.omx.observ.target.Species"},
		new String[] {"Individual", "ENTITY", "org.molgenis.omx.observ.target.Individual"},
		new String[] {"Panel", "ENTITY", "org.molgenis.omx.observ.target.Panel"},
		new String[] {"PanelSource", "ENTITY", "org.molgenis.omx.observ.target.PanelSource"},
		new String[] {"Ontology", "ENTITY", "org.molgenis.omx.observ.target.Ontology"},
		new String[] {"OntologyTerm", "ENTITY", "org.molgenis.omx.observ.target.OntologyTerm"},
		new String[] {"Accession", "ENTITY", "org.molgenis.omx.observ.target.Accession"},
		new String[] {"Value", "ENTITY", "org.molgenis.omx.observ.value.Value"},
		new String[] {"BoolValue", "ENTITY", "org.molgenis.omx.observ.value.BoolValue"},
		new String[] {"CategoricalValue", "ENTITY", "org.molgenis.omx.observ.value.CategoricalValue"},
		new String[] {"DateValue", "ENTITY", "org.molgenis.omx.observ.value.DateValue"},
		new String[] {"DateTimeValue", "ENTITY", "org.molgenis.omx.observ.value.DateTimeValue"},
		new String[] {"DecimalValue", "ENTITY", "org.molgenis.omx.observ.value.DecimalValue"},
		new String[] {"EmailValue", "ENTITY", "org.molgenis.omx.observ.value.EmailValue"},
		new String[] {"HtmlValue", "ENTITY", "org.molgenis.omx.observ.value.HtmlValue"},
		new String[] {"HyperlinkValue", "ENTITY", "org.molgenis.omx.observ.value.HyperlinkValue"},
		new String[] {"IntValue", "ENTITY", "org.molgenis.omx.observ.value.IntValue"},
		new String[] {"LongValue", "ENTITY", "org.molgenis.omx.observ.value.LongValue"},
		new String[] {"MrefValue", "ENTITY", "org.molgenis.omx.observ.value.MrefValue"},
		new String[] {"StringValue", "ENTITY", "org.molgenis.omx.observ.value.StringValue"},
		new String[] {"TextValue", "ENTITY", "org.molgenis.omx.observ.value.TextValue"},
		new String[] {"XrefValue", "ENTITY", "org.molgenis.omx.observ.value.XrefValue"},
		new String[] {"MolgenisRole", "ENTITY", "org.molgenis.omx.auth.MolgenisRole"},
		new String[] {"MolgenisGroup", "ENTITY", "org.molgenis.omx.auth.MolgenisGroup"},
		new String[] {"MolgenisRoleGroupLink", "ENTITY", "org.molgenis.omx.auth.MolgenisRoleGroupLink"},
		new String[] {"Person", "ENTITY", "org.molgenis.omx.auth.Person"},
		new String[] {"PersonRole", "ENTITY", "org.molgenis.omx.auth.PersonRole"},
		new String[] {"Institute", "ENTITY", "org.molgenis.omx.auth.Institute"},
		new String[] {"MolgenisUser", "ENTITY", "org.molgenis.omx.auth.MolgenisUser"},
		new String[] {"MolgenisPermission", "ENTITY", "org.molgenis.omx.auth.MolgenisPermission"},
		new String[] {"StudyDataRequest", "ENTITY", "org.molgenis.omx.study.StudyDataRequest"}
	};

	private static final String[][] UI_VALUES = new String[][] {
	};
}
