package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.CategoricalValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.CategoricalValueService;
import org.molgenis.omx.observ.Category;
import org.molgenis.controller.CategoryController.CategoryResponse;
import org.molgenis.service.CategoryService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/categoricalvalue")
public class CategoricalValueController
{
	@Autowired
	private CategoricalValueService categoricalValueService;

	@Autowired
	private CategoryService categoryService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<CategoricalValueResponse> createCategoricalValue(@Valid @RequestBody CategoricalValueRequest categoricalValueRequest)
			throws DatabaseException
	{
		return _createCategoricalValue(categoricalValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CategoricalValueResponse> createCategoricalValueFromForm(@Valid @ModelAttribute CategoricalValueRequest categoricalValueRequest)
			throws DatabaseException
	{
		return _createCategoricalValue(categoricalValueRequest);
	}

	private ResponseEntity<CategoricalValueResponse> _createCategoricalValue(CategoricalValueRequest categoricalValueRequest) throws DatabaseException
	{
		CategoricalValue categoricalValue = categoricalValueService.create(categoricalValueRequest.toCategoricalValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/categoricalvalue/" + categoricalValue.getId());
		return new ResponseEntity<CategoricalValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public CategoricalValueResponse retrieveCategoricalValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public CategoricalValueResponse retrieveCategoricalValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private CategoricalValueResponse _retrieveCategoricalValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		CategoricalValue categoricalValue = categoricalValueService.read(id);
		if (categoricalValue == null) throw new EntityNotFoundException("CategoricalValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new CategoricalValueResponse(categoricalValue, expandFields);
	}
			
	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET)
	public String retrieveCategoricalValueXrefValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValueXrefValue(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/value", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveCategoricalValueXrefValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValueXrefValue(id, "json", expandFields);
	}
	
	private String _retrieveCategoricalValueXrefValue(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		CategoricalValue categoricalValue = categoricalValueService.read(id);
		if (categoricalValue == null) throw new EntityNotFoundException("CategoricalValue " + id.toString() + " not found");
		Integer categoryId = categoricalValue.getValue_Id();
		String redirectUri = "redirect:/api/v1/category/" + categoryId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateCategoricalValue(@PathVariable Integer id, @Valid @RequestBody CategoricalValueRequest categoricalValueRequest)
			throws DatabaseException
	{
		_updateCategoricalValue(id, categoricalValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CategoricalValueResponse> updateCategoricalValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute CategoricalValueRequest categoricalValueRequest) throws DatabaseException
	{
		return _createCategoricalValue(categoricalValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCategoricalValuePost(@PathVariable Integer id, @Valid @RequestBody CategoricalValueRequest categoricalValueRequest)
			throws DatabaseException
	{
		_updateCategoricalValue(id, categoricalValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCategoricalValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute CategoricalValueRequest categoricalValueRequest)
			throws DatabaseException
	{
		_updateCategoricalValue(id, categoricalValueRequest);
	}

	private void _updateCategoricalValue(Integer id, CategoricalValueRequest categoricalValueRequest) throws DatabaseException
	{
		CategoricalValue categoricalValue = categoricalValueRequest.toCategoricalValue();
		categoricalValue.setId(id);
		categoricalValueService.update(categoricalValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCategoricalValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCategoricalValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCategoricalValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCategoricalValue(id);
	}

	private void _deleteCategoricalValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = categoricalValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("CategoricalValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<CategoricalValueResponse> retrieveCategoricalValueCollection(@Valid EntityCollectionRequest categoricalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValueCollection(categoricalValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CategoricalValueResponse> retrieveCategoricalValueCollectionJson(@Valid EntityCollectionRequest categoricalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValueCollection(categoricalValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<CategoricalValueResponse> retrieveCategoricalValueCollectionPost(@Valid @RequestBody EntityCollectionRequest categoricalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValueCollection(categoricalValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CategoricalValueResponse> retrieveCategoricalValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest categoricalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoricalValueCollection(categoricalValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<CategoricalValueResponse> _retrieveCategoricalValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<CategoricalValue> categoricalValuePager = categoricalValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<CategoricalValueResponse>(categoricalValuePager, Lists.newArrayList(Iterables.transform(
				categoricalValuePager.getIterable(), new Function<CategoricalValue, CategoricalValueResponse>()
				{
					@Override
					@Nullable
					public CategoricalValueResponse apply(@Nullable CategoricalValue categoricalValue)
					{
						try
						{
							return categoricalValue != null ? new CategoricalValueResponse(categoricalValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/categoricalvalue");
	}

	private static class CategoricalValueRequest
	{
		private Integer value;
	
		public CategoricalValue toCategoricalValue()
		{
			CategoricalValue categoricalValue = new CategoricalValue();
			categoricalValue.setValue_Id(value);
			return categoricalValue;
		}
		
		public void setValue(Integer value)
		{
			this.value = value;
		}
		
	}

	static class CategoricalValueResponse
	{
		private final String href;
		private final Object value;
	
		public CategoricalValueResponse(CategoricalValue categoricalValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/categoricalvalue/" + categoricalValue.getId();
			if (expandFields != null && expandFields.contains("value")) this.value = new CategoryResponse(categoricalValue.getValue(), null);
			else this.value = java.util.Collections.singletonMap("href", "/api/v1/categoricalvalue/" + categoricalValue.getId() + "/value");
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public Object getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}