package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.DateTimeValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.DateTimeValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/datetimevalue")
public class DateTimeValueController
{
	@Autowired
	private DateTimeValueService dateTimeValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<DateTimeValueResponse> createDateTimeValue(@Valid @RequestBody DateTimeValueRequest dateTimeValueRequest)
			throws DatabaseException
	{
		return _createDateTimeValue(dateTimeValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DateTimeValueResponse> createDateTimeValueFromForm(@Valid @ModelAttribute DateTimeValueRequest dateTimeValueRequest)
			throws DatabaseException
	{
		return _createDateTimeValue(dateTimeValueRequest);
	}

	private ResponseEntity<DateTimeValueResponse> _createDateTimeValue(DateTimeValueRequest dateTimeValueRequest) throws DatabaseException
	{
		DateTimeValue dateTimeValue = dateTimeValueService.create(dateTimeValueRequest.toDateTimeValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/datetimevalue/" + dateTimeValue.getId());
		return new ResponseEntity<DateTimeValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public DateTimeValueResponse retrieveDateTimeValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateTimeValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public DateTimeValueResponse retrieveDateTimeValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateTimeValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private DateTimeValueResponse _retrieveDateTimeValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		DateTimeValue dateTimeValue = dateTimeValueService.read(id);
		if (dateTimeValue == null) throw new EntityNotFoundException("DateTimeValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new DateTimeValueResponse(dateTimeValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateDateTimeValue(@PathVariable Integer id, @Valid @RequestBody DateTimeValueRequest dateTimeValueRequest)
			throws DatabaseException
	{
		_updateDateTimeValue(id, dateTimeValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DateTimeValueResponse> updateDateTimeValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute DateTimeValueRequest dateTimeValueRequest) throws DatabaseException
	{
		return _createDateTimeValue(dateTimeValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDateTimeValuePost(@PathVariable Integer id, @Valid @RequestBody DateTimeValueRequest dateTimeValueRequest)
			throws DatabaseException
	{
		_updateDateTimeValue(id, dateTimeValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDateTimeValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute DateTimeValueRequest dateTimeValueRequest)
			throws DatabaseException
	{
		_updateDateTimeValue(id, dateTimeValueRequest);
	}

	private void _updateDateTimeValue(Integer id, DateTimeValueRequest dateTimeValueRequest) throws DatabaseException
	{
		DateTimeValue dateTimeValue = dateTimeValueRequest.toDateTimeValue();
		dateTimeValue.setId(id);
		dateTimeValueService.update(dateTimeValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDateTimeValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDateTimeValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDateTimeValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDateTimeValue(id);
	}

	private void _deleteDateTimeValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = dateTimeValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("DateTimeValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<DateTimeValueResponse> retrieveDateTimeValueCollection(@Valid EntityCollectionRequest dateTimeValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateTimeValueCollection(dateTimeValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DateTimeValueResponse> retrieveDateTimeValueCollectionJson(@Valid EntityCollectionRequest dateTimeValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateTimeValueCollection(dateTimeValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<DateTimeValueResponse> retrieveDateTimeValueCollectionPost(@Valid @RequestBody EntityCollectionRequest dateTimeValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateTimeValueCollection(dateTimeValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DateTimeValueResponse> retrieveDateTimeValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest dateTimeValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateTimeValueCollection(dateTimeValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<DateTimeValueResponse> _retrieveDateTimeValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<DateTimeValue> dateTimeValuePager = dateTimeValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<DateTimeValueResponse>(dateTimeValuePager, Lists.newArrayList(Iterables.transform(
				dateTimeValuePager.getIterable(), new Function<DateTimeValue, DateTimeValueResponse>()
				{
					@Override
					@Nullable
					public DateTimeValueResponse apply(@Nullable DateTimeValue dateTimeValue)
					{
						try
						{
							return dateTimeValue != null ? new DateTimeValueResponse(dateTimeValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/datetimevalue");
	}

	private static class DateTimeValueRequest
	{
		private java.util.Date value;
	
		public DateTimeValue toDateTimeValue()
		{
			DateTimeValue dateTimeValue = new DateTimeValue();
			dateTimeValue.setValue(value);
			return dateTimeValue;
		}
		
		public void setValue(java.util.Date value)
		{
			this.value = value;
		}
		
	}

	static class DateTimeValueResponse
	{
		private final String href;
		private final java.util.Date value;
	
		public DateTimeValueResponse(DateTimeValue dateTimeValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/datetimevalue/" + dateTimeValue.getId();
			this.value = dateTimeValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public java.util.Date getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}