package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.HyperlinkValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.HyperlinkValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/hyperlinkvalue")
public class HyperlinkValueController
{
	@Autowired
	private HyperlinkValueService hyperlinkValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<HyperlinkValueResponse> createHyperlinkValue(@Valid @RequestBody HyperlinkValueRequest hyperlinkValueRequest)
			throws DatabaseException
	{
		return _createHyperlinkValue(hyperlinkValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<HyperlinkValueResponse> createHyperlinkValueFromForm(@Valid @ModelAttribute HyperlinkValueRequest hyperlinkValueRequest)
			throws DatabaseException
	{
		return _createHyperlinkValue(hyperlinkValueRequest);
	}

	private ResponseEntity<HyperlinkValueResponse> _createHyperlinkValue(HyperlinkValueRequest hyperlinkValueRequest) throws DatabaseException
	{
		HyperlinkValue hyperlinkValue = hyperlinkValueService.create(hyperlinkValueRequest.toHyperlinkValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/hyperlinkvalue/" + hyperlinkValue.getId());
		return new ResponseEntity<HyperlinkValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public HyperlinkValueResponse retrieveHyperlinkValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHyperlinkValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public HyperlinkValueResponse retrieveHyperlinkValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHyperlinkValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private HyperlinkValueResponse _retrieveHyperlinkValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		HyperlinkValue hyperlinkValue = hyperlinkValueService.read(id);
		if (hyperlinkValue == null) throw new EntityNotFoundException("HyperlinkValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new HyperlinkValueResponse(hyperlinkValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateHyperlinkValue(@PathVariable Integer id, @Valid @RequestBody HyperlinkValueRequest hyperlinkValueRequest)
			throws DatabaseException
	{
		_updateHyperlinkValue(id, hyperlinkValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<HyperlinkValueResponse> updateHyperlinkValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute HyperlinkValueRequest hyperlinkValueRequest) throws DatabaseException
	{
		return _createHyperlinkValue(hyperlinkValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateHyperlinkValuePost(@PathVariable Integer id, @Valid @RequestBody HyperlinkValueRequest hyperlinkValueRequest)
			throws DatabaseException
	{
		_updateHyperlinkValue(id, hyperlinkValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateHyperlinkValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute HyperlinkValueRequest hyperlinkValueRequest)
			throws DatabaseException
	{
		_updateHyperlinkValue(id, hyperlinkValueRequest);
	}

	private void _updateHyperlinkValue(Integer id, HyperlinkValueRequest hyperlinkValueRequest) throws DatabaseException
	{
		HyperlinkValue hyperlinkValue = hyperlinkValueRequest.toHyperlinkValue();
		hyperlinkValue.setId(id);
		hyperlinkValueService.update(hyperlinkValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteHyperlinkValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteHyperlinkValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteHyperlinkValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteHyperlinkValue(id);
	}

	private void _deleteHyperlinkValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = hyperlinkValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("HyperlinkValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<HyperlinkValueResponse> retrieveHyperlinkValueCollection(@Valid EntityCollectionRequest hyperlinkValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHyperlinkValueCollection(hyperlinkValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<HyperlinkValueResponse> retrieveHyperlinkValueCollectionJson(@Valid EntityCollectionRequest hyperlinkValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHyperlinkValueCollection(hyperlinkValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<HyperlinkValueResponse> retrieveHyperlinkValueCollectionPost(@Valid @RequestBody EntityCollectionRequest hyperlinkValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHyperlinkValueCollection(hyperlinkValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<HyperlinkValueResponse> retrieveHyperlinkValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest hyperlinkValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHyperlinkValueCollection(hyperlinkValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<HyperlinkValueResponse> _retrieveHyperlinkValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<HyperlinkValue> hyperlinkValuePager = hyperlinkValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<HyperlinkValueResponse>(hyperlinkValuePager, Lists.newArrayList(Iterables.transform(
				hyperlinkValuePager.getIterable(), new Function<HyperlinkValue, HyperlinkValueResponse>()
				{
					@Override
					@Nullable
					public HyperlinkValueResponse apply(@Nullable HyperlinkValue hyperlinkValue)
					{
						try
						{
							return hyperlinkValue != null ? new HyperlinkValueResponse(hyperlinkValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/hyperlinkvalue");
	}

	private static class HyperlinkValueRequest
	{
		private String value;
	
		public HyperlinkValue toHyperlinkValue()
		{
			HyperlinkValue hyperlinkValue = new HyperlinkValue();
			hyperlinkValue.setValue(value);
			return hyperlinkValue;
		}
		
		public void setValue(String value)
		{
			this.value = value;
		}
		
	}

	static class HyperlinkValueResponse
	{
		private final String href;
		private final String value;
	
		public HyperlinkValueResponse(HyperlinkValue hyperlinkValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/hyperlinkvalue/" + hyperlinkValue.getId();
			this.value = hyperlinkValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public String getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}