package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.ObservationTarget;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ObservationTargetService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/observationtarget")
public class ObservationTargetController
{
	@Autowired
	private ObservationTargetService observationTargetService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ObservationTargetResponse> createObservationTarget(@Valid @RequestBody ObservationTargetRequest observationTargetRequest)
			throws DatabaseException
	{
		return _createObservationTarget(observationTargetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservationTargetResponse> createObservationTargetFromForm(@Valid @ModelAttribute ObservationTargetRequest observationTargetRequest)
			throws DatabaseException
	{
		return _createObservationTarget(observationTargetRequest);
	}

	private ResponseEntity<ObservationTargetResponse> _createObservationTarget(ObservationTargetRequest observationTargetRequest) throws DatabaseException
	{
		ObservationTarget observationTarget = observationTargetService.create(observationTargetRequest.toObservationTarget());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/observationtarget/" + observationTarget.getId());
		return new ResponseEntity<ObservationTargetResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ObservationTargetResponse retrieveObservationTarget(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationTarget(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ObservationTargetResponse retrieveObservationTargetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationTarget(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ObservationTargetResponse _retrieveObservationTarget(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		ObservationTarget observationTarget = observationTargetService.read(id);
		if (observationTarget == null) throw new EntityNotFoundException("ObservationTarget " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ObservationTargetResponse(observationTarget, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateObservationTarget(@PathVariable Integer id, @Valid @RequestBody ObservationTargetRequest observationTargetRequest)
			throws DatabaseException
	{
		_updateObservationTarget(id, observationTargetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservationTargetResponse> updateObservationTargetFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ObservationTargetRequest observationTargetRequest) throws DatabaseException
	{
		return _createObservationTarget(observationTargetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservationTargetPost(@PathVariable Integer id, @Valid @RequestBody ObservationTargetRequest observationTargetRequest)
			throws DatabaseException
	{
		_updateObservationTarget(id, observationTargetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservationTargetFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ObservationTargetRequest observationTargetRequest)
			throws DatabaseException
	{
		_updateObservationTarget(id, observationTargetRequest);
	}

	private void _updateObservationTarget(Integer id, ObservationTargetRequest observationTargetRequest) throws DatabaseException
	{
		ObservationTarget observationTarget = observationTargetRequest.toObservationTarget();
		observationTarget.setId(id);
		observationTargetService.update(observationTarget);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservationTarget(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservationTarget(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservationTargetPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservationTarget(id);
	}

	private void _deleteObservationTarget(Integer id) throws DatabaseException
	{
		boolean isDeleted = observationTargetService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("ObservationTarget " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ObservationTargetResponse> retrieveObservationTargetCollection(@Valid EntityCollectionRequest observationTargetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationTargetCollection(observationTargetCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservationTargetResponse> retrieveObservationTargetCollectionJson(@Valid EntityCollectionRequest observationTargetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationTargetCollection(observationTargetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ObservationTargetResponse> retrieveObservationTargetCollectionPost(@Valid @RequestBody EntityCollectionRequest observationTargetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationTargetCollection(observationTargetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservationTargetResponse> retrieveObservationTargetCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest observationTargetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationTargetCollection(observationTargetCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ObservationTargetResponse> _retrieveObservationTargetCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<ObservationTarget> observationTargetPager = observationTargetService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ObservationTargetResponse>(observationTargetPager, Lists.newArrayList(Iterables.transform(
				observationTargetPager.getIterable(), new Function<ObservationTarget, ObservationTargetResponse>()
				{
					@Override
					@Nullable
					public ObservationTargetResponse apply(@Nullable ObservationTarget observationTarget)
					{
						try
						{
							return observationTarget != null ? new ObservationTargetResponse(observationTarget, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/observationtarget");
	}

	private static class ObservationTargetRequest
	{
		private String identifier;
		private String name;
		private String description;
	
		public ObservationTarget toObservationTarget()
		{
			ObservationTarget observationTarget = new ObservationTarget();
			observationTarget.setIdentifier(identifier);
			observationTarget.setName(name);
			observationTarget.setDescription(description);
			return observationTarget;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
	}

	static class ObservationTargetResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
	
		public ObservationTargetResponse(ObservationTarget observationTarget, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/observationtarget/" + observationTarget.getId();
			this.identifier = observationTarget.getIdentifier();
			this.name = observationTarget.getName();
			this.description = observationTarget.getDescription();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}