package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.target.PanelSource;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.PanelSourceService;
import org.molgenis.omx.observ.target.Panel;
import org.molgenis.controller.PanelController.PanelResponse;
import org.molgenis.service.PanelService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/panelsource")
public class PanelSourceController
{
	@Autowired
	private PanelSourceService panelSourceService;

	@Autowired
	private PanelService panelService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<PanelSourceResponse> createPanelSource(@Valid @RequestBody PanelSourceRequest panelSourceRequest)
			throws DatabaseException
	{
		return _createPanelSource(panelSourceRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PanelSourceResponse> createPanelSourceFromForm(@Valid @ModelAttribute PanelSourceRequest panelSourceRequest)
			throws DatabaseException
	{
		return _createPanelSource(panelSourceRequest);
	}

	private ResponseEntity<PanelSourceResponse> _createPanelSource(PanelSourceRequest panelSourceRequest) throws DatabaseException
	{
		PanelSource panelSource = panelSourceService.create(panelSourceRequest.toPanelSource());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/panelsource/" + panelSource.getId());
		return new ResponseEntity<PanelSourceResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public PanelSourceResponse retrievePanelSource(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSource(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public PanelSourceResponse retrievePanelSourceJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSource(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private PanelSourceResponse _retrievePanelSource(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		PanelSource panelSource = panelSourceService.read(id);
		if (panelSource == null) throw new EntityNotFoundException("PanelSource " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new PanelSourceResponse(panelSource, expandFields);
	}
			
	@RequestMapping(value = "/{id}/currentPanel", method = RequestMethod.GET)
	public String retrievePanelSourceXrefCurrentPanel(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceXrefCurrentPanel(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/currentPanel", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePanelSourceXrefCurrentPanelJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceXrefCurrentPanel(id, "json", expandFields);
	}
	
	private String _retrievePanelSourceXrefCurrentPanel(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		PanelSource panelSource = panelSourceService.read(id);
		if (panelSource == null) throw new EntityNotFoundException("PanelSource " + id.toString() + " not found");
		Integer panelId = panelSource.getCurrentPanel_Id();
		String redirectUri = "redirect:/api/v1/panel/" + panelId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/sourcePanel", method = RequestMethod.GET)
	public String retrievePanelSourceXrefSourcePanel(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceXrefSourcePanel(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/sourcePanel", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePanelSourceXrefSourcePanelJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceXrefSourcePanel(id, "json", expandFields);
	}
	
	private String _retrievePanelSourceXrefSourcePanel(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		PanelSource panelSource = panelSourceService.read(id);
		if (panelSource == null) throw new EntityNotFoundException("PanelSource " + id.toString() + " not found");
		Integer panelId = panelSource.getSourcePanel_Id();
		String redirectUri = "redirect:/api/v1/panel/" + panelId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updatePanelSource(@PathVariable Integer id, @Valid @RequestBody PanelSourceRequest panelSourceRequest)
			throws DatabaseException
	{
		_updatePanelSource(id, panelSourceRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PanelSourceResponse> updatePanelSourceFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute PanelSourceRequest panelSourceRequest) throws DatabaseException
	{
		return _createPanelSource(panelSourceRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePanelSourcePost(@PathVariable Integer id, @Valid @RequestBody PanelSourceRequest panelSourceRequest)
			throws DatabaseException
	{
		_updatePanelSource(id, panelSourceRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePanelSourceFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute PanelSourceRequest panelSourceRequest)
			throws DatabaseException
	{
		_updatePanelSource(id, panelSourceRequest);
	}

	private void _updatePanelSource(Integer id, PanelSourceRequest panelSourceRequest) throws DatabaseException
	{
		PanelSource panelSource = panelSourceRequest.toPanelSource();
		panelSource.setId(id);
		panelSourceService.update(panelSource);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePanelSource(@PathVariable Integer id) throws DatabaseException
	{
		_deletePanelSource(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePanelSourcePost(@PathVariable Integer id) throws DatabaseException
	{
		_deletePanelSource(id);
	}

	private void _deletePanelSource(Integer id) throws DatabaseException
	{
		boolean isDeleted = panelSourceService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("PanelSource " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PanelSourceResponse> retrievePanelSourceCollection(@Valid EntityCollectionRequest panelSourceCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceCollection(panelSourceCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PanelSourceResponse> retrievePanelSourceCollectionJson(@Valid EntityCollectionRequest panelSourceCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceCollection(panelSourceCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<PanelSourceResponse> retrievePanelSourceCollectionPost(@Valid @RequestBody EntityCollectionRequest panelSourceCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceCollection(panelSourceCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PanelSourceResponse> retrievePanelSourceCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest panelSourceCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePanelSourceCollection(panelSourceCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<PanelSourceResponse> _retrievePanelSourceCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<PanelSource> panelSourcePager = panelSourceService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<PanelSourceResponse>(panelSourcePager, Lists.newArrayList(Iterables.transform(
				panelSourcePager.getIterable(), new Function<PanelSource, PanelSourceResponse>()
				{
					@Override
					@Nullable
					public PanelSourceResponse apply(@Nullable PanelSource panelSource)
					{
						try
						{
							return panelSource != null ? new PanelSourceResponse(panelSource, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/panelsource");
	}

	private static class PanelSourceRequest
	{
		private Integer currentPanel;
		private Integer sourcePanel;
		private Integer numberOfIndividuals;
		private String selectionCriteria;
	
		public PanelSource toPanelSource()
		{
			PanelSource panelSource = new PanelSource();
			panelSource.setCurrentPanel_Id(currentPanel);
			panelSource.setSourcePanel_Id(sourcePanel);
			panelSource.setNumberOfIndividuals(numberOfIndividuals);
			panelSource.setSelectionCriteria(selectionCriteria);
			return panelSource;
		}
		
		public void setCurrentPanel(Integer currentPanel)
		{
			this.currentPanel = currentPanel;
		}
		
		public void setSourcePanel(Integer sourcePanel)
		{
			this.sourcePanel = sourcePanel;
		}
		
		public void setNumberOfIndividuals(Integer numberOfIndividuals)
		{
			this.numberOfIndividuals = numberOfIndividuals;
		}
		
		public void setSelectionCriteria(String selectionCriteria)
		{
			this.selectionCriteria = selectionCriteria;
		}
		
	}

	static class PanelSourceResponse
	{
		private final String href;
		private final Object currentPanel;
		private final Object sourcePanel;
		private final Integer numberOfIndividuals;
		private final String selectionCriteria;
	
		public PanelSourceResponse(PanelSource panelSource, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/panelsource/" + panelSource.getId();
			if (expandFields != null && expandFields.contains("currentPanel")) this.currentPanel = new PanelResponse(panelSource.getCurrentPanel(), null);
			else this.currentPanel = java.util.Collections.singletonMap("href", "/api/v1/panelsource/" + panelSource.getId() + "/currentPanel");
			if (expandFields != null && expandFields.contains("sourcePanel")) this.sourcePanel = new PanelResponse(panelSource.getSourcePanel(), null);
			else this.sourcePanel = java.util.Collections.singletonMap("href", "/api/v1/panelsource/" + panelSource.getId() + "/sourcePanel");
			this.numberOfIndividuals = panelSource.getNumberOfIndividuals();
			this.selectionCriteria = panelSource.getSelectionCriteria();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public Object getCurrentPanel()
		{
			return currentPanel;
		}
	
		public Object getSourcePanel()
		{
			return sourcePanel;
		}
	
		public Integer getNumberOfIndividuals()
		{
			return numberOfIndividuals;
		}
	
		public String getSelectionCriteria()
		{
			return selectionCriteria;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}