package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.Protocol;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ProtocolService;
import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.controller.OntologyTermController.OntologyTermResponse;
import org.molgenis.service.OntologyTermService;
import org.molgenis.omx.observ.ObservableFeature;
import org.molgenis.controller.ObservableFeatureController.ObservableFeatureResponse;
import org.molgenis.service.ObservableFeatureService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/protocol")
public class ProtocolController
{
	@Autowired
	private ProtocolService protocolService;

	@Autowired
	private OntologyTermService ontologyTermService;
		
	@Autowired
	private ObservableFeatureService observableFeatureService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ProtocolResponse> createProtocol(@Valid @RequestBody ProtocolRequest protocolRequest)
			throws DatabaseException
	{
		return _createProtocol(protocolRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProtocolResponse> createProtocolFromForm(@Valid @ModelAttribute ProtocolRequest protocolRequest)
			throws DatabaseException
	{
		return _createProtocol(protocolRequest);
	}

	private ResponseEntity<ProtocolResponse> _createProtocol(ProtocolRequest protocolRequest) throws DatabaseException
	{
		Protocol protocol = protocolService.create(protocolRequest.toProtocol());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/protocol/" + protocol.getId());
		return new ResponseEntity<ProtocolResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ProtocolResponse retrieveProtocol(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocol(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ProtocolResponse retrieveProtocolJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocol(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ProtocolResponse _retrieveProtocol(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Protocol protocol = protocolService.read(id);
		if (protocol == null) throw new EntityNotFoundException("Protocol " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ProtocolResponse(protocol, expandFields);
	}
			
	@RequestMapping(value = "/{id}/protocolType", method = RequestMethod.GET)
	public String retrieveProtocolXrefProtocolType(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolXrefProtocolType(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/protocolType", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProtocolXrefProtocolTypeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolXrefProtocolType(id, "json", expandFields);
	}
	
	private String _retrieveProtocolXrefProtocolType(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Protocol protocol = protocolService.read(id);
		if (protocol == null) throw new EntityNotFoundException("Protocol " + id.toString() + " not found");
		Integer ontologyTermId = protocol.getProtocolType_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/subprotocols", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ProtocolResponse> retrieveProtocolMrefSubprotocols(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Protocol protocol = protocolService.read(id);
		if (protocol == null) throw new EntityNotFoundException("Protocol " + id.toString() + " not found");
		return _retrieveProtocolMrefSubprotocols(protocol, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/subprotocols", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProtocolResponse> retrieveProtocolMrefSubprotocolsJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Protocol protocol = protocolService.read(id);
		if (protocol == null) throw new EntityNotFoundException("Protocol " + id.toString() + " not found");
		return _retrieveProtocolMrefSubprotocols(protocol, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<ProtocolResponse> _retrieveProtocolMrefSubprotocols(Protocol protocol, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<Protocol> protocolCollection = protocol.getSubprotocols();
		
		int total = protocolCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		protocolCollection = protocolCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<Protocol> protocolPager = new EntityPager<Protocol>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, protocolCollection);
		
		return new EntityCollectionResponse<ProtocolResponse>(protocolPager, Lists.newArrayList(Iterables.transform(protocolCollection,
				new Function<Protocol, ProtocolResponse>()
				{
					@Override
					@Nullable
					public ProtocolResponse apply(@Nullable Protocol protocol)
					{
						try
						{
							return protocol != null ? new ProtocolResponse(protocol, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/protocol/" + protocol.getId() + "/subprotocols");
	}
	@RequestMapping(value = "/{id}/features", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveProtocolMrefFeatures(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Protocol protocol = protocolService.read(id);
		if (protocol == null) throw new EntityNotFoundException("Protocol " + id.toString() + " not found");
		return _retrieveProtocolMrefFeatures(protocol, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/features", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveProtocolMrefFeaturesJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Protocol protocol = protocolService.read(id);
		if (protocol == null) throw new EntityNotFoundException("Protocol " + id.toString() + " not found");
		return _retrieveProtocolMrefFeatures(protocol, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<ObservableFeatureResponse> _retrieveProtocolMrefFeatures(Protocol protocol, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<ObservableFeature> observableFeatureCollection = protocol.getFeatures();
		
		int total = observableFeatureCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		observableFeatureCollection = observableFeatureCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<ObservableFeature> observableFeaturePager = new EntityPager<ObservableFeature>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, observableFeatureCollection);
		
		return new EntityCollectionResponse<ObservableFeatureResponse>(observableFeaturePager, Lists.newArrayList(Iterables.transform(observableFeatureCollection,
				new Function<ObservableFeature, ObservableFeatureResponse>()
				{
					@Override
					@Nullable
					public ObservableFeatureResponse apply(@Nullable ObservableFeature observableFeature)
					{
						try
						{
							return observableFeature != null ? new ObservableFeatureResponse(observableFeature, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/protocol/" + protocol.getId() + "/features");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateProtocol(@PathVariable Integer id, @Valid @RequestBody ProtocolRequest protocolRequest)
			throws DatabaseException
	{
		_updateProtocol(id, protocolRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProtocolResponse> updateProtocolFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ProtocolRequest protocolRequest) throws DatabaseException
	{
		return _createProtocol(protocolRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProtocolPost(@PathVariable Integer id, @Valid @RequestBody ProtocolRequest protocolRequest)
			throws DatabaseException
	{
		_updateProtocol(id, protocolRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProtocolFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ProtocolRequest protocolRequest)
			throws DatabaseException
	{
		_updateProtocol(id, protocolRequest);
	}

	private void _updateProtocol(Integer id, ProtocolRequest protocolRequest) throws DatabaseException
	{
		Protocol protocol = protocolRequest.toProtocol();
		protocol.setId(id);
		protocolService.update(protocol);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProtocol(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProtocol(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProtocolPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProtocol(id);
	}

	private void _deleteProtocol(Integer id) throws DatabaseException
	{
		boolean isDeleted = protocolService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Protocol " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ProtocolResponse> retrieveProtocolCollection(@Valid EntityCollectionRequest protocolCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolCollection(protocolCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProtocolResponse> retrieveProtocolCollectionJson(@Valid EntityCollectionRequest protocolCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolCollection(protocolCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ProtocolResponse> retrieveProtocolCollectionPost(@Valid @RequestBody EntityCollectionRequest protocolCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolCollection(protocolCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProtocolResponse> retrieveProtocolCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest protocolCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolCollection(protocolCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ProtocolResponse> _retrieveProtocolCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Protocol> protocolPager = protocolService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ProtocolResponse>(protocolPager, Lists.newArrayList(Iterables.transform(
				protocolPager.getIterable(), new Function<Protocol, ProtocolResponse>()
				{
					@Override
					@Nullable
					public ProtocolResponse apply(@Nullable Protocol protocol)
					{
						try
						{
							return protocol != null ? new ProtocolResponse(protocol, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/protocol");
	}

	private static class ProtocolRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer protocolType;
		private java.util.List<Integer> subprotocols;
		private java.util.List<Integer> features;
	
		public Protocol toProtocol()
		{
			Protocol protocol = new Protocol();
			protocol.setIdentifier(identifier);
			protocol.setName(name);
			protocol.setDescription(description);
			protocol.setProtocolType_Id(protocolType);
			protocol.setSubprotocols_Id(subprotocols);
			protocol.setFeatures_Id(features);
			return protocol;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setProtocolType(Integer protocolType)
		{
			this.protocolType = protocolType;
		}
		
		public void setSubprotocols(java.util.List<Integer> subprotocols)
		{
			this.subprotocols = subprotocols;
		}
		
		public void setFeatures(java.util.List<Integer> features)
		{
			this.features = features;
		}
		
	}

	static class ProtocolResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object protocolType;
		private final Object subprotocols;
		private final Object features;
	
		public ProtocolResponse(Protocol protocol, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/protocol/" + protocol.getId();
			this.identifier = protocol.getIdentifier();
			this.name = protocol.getName();
			this.description = protocol.getDescription();
			if (expandFields != null && expandFields.contains("protocolType")) this.protocolType = protocol.getProtocolType() == null ? null : new OntologyTermResponse(protocol.getProtocolType(), null);
			else this.protocolType = protocol.getProtocolType() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/protocol/" + protocol.getId() + "/protocolType");
			java.util.List<Protocol> subprotocolsCollection = protocol.getSubprotocols();
			if (expandFields != null && expandFields.contains("subprotocols")) this.subprotocols = subprotocolsCollection == null ? null : _retrieveProtocolMrefSubprotocols(protocol, new EntityCollectionRequest());
			else this.subprotocols = subprotocolsCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/protocol/" + protocol.getId() + "/subprotocols");
			java.util.List<ObservableFeature> FeaturesCollection = protocol.getFeatures();
			if (expandFields != null && expandFields.contains("features")) this.features = FeaturesCollection == null ? null : _retrieveProtocolMrefFeatures(protocol, new EntityCollectionRequest());
			else this.features = FeaturesCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/protocol/" + protocol.getId() + "/features");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getProtocolType()
		{
			return protocolType;
		}
	
		public Object getSubprotocols()
		{
			return subprotocols;
		}
	
		public Object getFeatures()
		{
			return features;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}