package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.core.RuntimeProperty;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.RuntimePropertyService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/runtimeproperty")
public class RuntimePropertyController
{
	@Autowired
	private RuntimePropertyService runtimePropertyService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<RuntimePropertyResponse> createRuntimeProperty(@Valid @RequestBody RuntimePropertyRequest runtimePropertyRequest)
			throws DatabaseException
	{
		return _createRuntimeProperty(runtimePropertyRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<RuntimePropertyResponse> createRuntimePropertyFromForm(@Valid @ModelAttribute RuntimePropertyRequest runtimePropertyRequest)
			throws DatabaseException
	{
		return _createRuntimeProperty(runtimePropertyRequest);
	}

	private ResponseEntity<RuntimePropertyResponse> _createRuntimeProperty(RuntimePropertyRequest runtimePropertyRequest) throws DatabaseException
	{
		RuntimeProperty runtimeProperty = runtimePropertyService.create(runtimePropertyRequest.toRuntimeProperty());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/runtimeproperty/" + runtimeProperty.getId());
		return new ResponseEntity<RuntimePropertyResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public RuntimePropertyResponse retrieveRuntimeProperty(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveRuntimeProperty(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public RuntimePropertyResponse retrieveRuntimePropertyJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveRuntimeProperty(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private RuntimePropertyResponse _retrieveRuntimeProperty(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		RuntimeProperty runtimeProperty = runtimePropertyService.read(id);
		if (runtimeProperty == null) throw new EntityNotFoundException("RuntimeProperty " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new RuntimePropertyResponse(runtimeProperty, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateRuntimeProperty(@PathVariable Integer id, @Valid @RequestBody RuntimePropertyRequest runtimePropertyRequest)
			throws DatabaseException
	{
		_updateRuntimeProperty(id, runtimePropertyRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<RuntimePropertyResponse> updateRuntimePropertyFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute RuntimePropertyRequest runtimePropertyRequest) throws DatabaseException
	{
		return _createRuntimeProperty(runtimePropertyRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateRuntimePropertyPost(@PathVariable Integer id, @Valid @RequestBody RuntimePropertyRequest runtimePropertyRequest)
			throws DatabaseException
	{
		_updateRuntimeProperty(id, runtimePropertyRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateRuntimePropertyFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute RuntimePropertyRequest runtimePropertyRequest)
			throws DatabaseException
	{
		_updateRuntimeProperty(id, runtimePropertyRequest);
	}

	private void _updateRuntimeProperty(Integer id, RuntimePropertyRequest runtimePropertyRequest) throws DatabaseException
	{
		RuntimeProperty runtimeProperty = runtimePropertyRequest.toRuntimeProperty();
		runtimeProperty.setId(id);
		runtimePropertyService.update(runtimeProperty);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteRuntimeProperty(@PathVariable Integer id) throws DatabaseException
	{
		_deleteRuntimeProperty(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteRuntimePropertyPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteRuntimeProperty(id);
	}

	private void _deleteRuntimeProperty(Integer id) throws DatabaseException
	{
		boolean isDeleted = runtimePropertyService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("RuntimeProperty " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<RuntimePropertyResponse> retrieveRuntimePropertyCollection(@Valid EntityCollectionRequest runtimePropertyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveRuntimePropertyCollection(runtimePropertyCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<RuntimePropertyResponse> retrieveRuntimePropertyCollectionJson(@Valid EntityCollectionRequest runtimePropertyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveRuntimePropertyCollection(runtimePropertyCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<RuntimePropertyResponse> retrieveRuntimePropertyCollectionPost(@Valid @RequestBody EntityCollectionRequest runtimePropertyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveRuntimePropertyCollection(runtimePropertyCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<RuntimePropertyResponse> retrieveRuntimePropertyCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest runtimePropertyCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveRuntimePropertyCollection(runtimePropertyCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<RuntimePropertyResponse> _retrieveRuntimePropertyCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<RuntimeProperty> runtimePropertyPager = runtimePropertyService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<RuntimePropertyResponse>(runtimePropertyPager, Lists.newArrayList(Iterables.transform(
				runtimePropertyPager.getIterable(), new Function<RuntimeProperty, RuntimePropertyResponse>()
				{
					@Override
					@Nullable
					public RuntimePropertyResponse apply(@Nullable RuntimeProperty runtimeProperty)
					{
						try
						{
							return runtimeProperty != null ? new RuntimePropertyResponse(runtimeProperty, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/runtimeproperty");
	}

	private static class RuntimePropertyRequest
	{
		private String identifier;
		private String name;
		private String value;
	
		public RuntimeProperty toRuntimeProperty()
		{
			RuntimeProperty runtimeProperty = new RuntimeProperty();
			runtimeProperty.setIdentifier(identifier);
			runtimeProperty.setName(name);
			runtimeProperty.setValue(value);
			return runtimeProperty;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setValue(String value)
		{
			this.value = value;
		}
		
	}

	static class RuntimePropertyResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String value;
	
		public RuntimePropertyResponse(RuntimeProperty runtimeProperty, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/runtimeproperty/" + runtimeProperty.getId();
			this.identifier = runtimeProperty.getIdentifier();
			this.name = runtimeProperty.getName();
			this.value = runtimeProperty.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
		public String getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}