package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.StringValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.StringValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/stringvalue")
public class StringValueController
{
	@Autowired
	private StringValueService stringValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<StringValueResponse> createStringValue(@Valid @RequestBody StringValueRequest stringValueRequest)
			throws DatabaseException
	{
		return _createStringValue(stringValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<StringValueResponse> createStringValueFromForm(@Valid @ModelAttribute StringValueRequest stringValueRequest)
			throws DatabaseException
	{
		return _createStringValue(stringValueRequest);
	}

	private ResponseEntity<StringValueResponse> _createStringValue(StringValueRequest stringValueRequest) throws DatabaseException
	{
		StringValue stringValue = stringValueService.create(stringValueRequest.toStringValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/stringvalue/" + stringValue.getId());
		return new ResponseEntity<StringValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public StringValueResponse retrieveStringValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStringValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public StringValueResponse retrieveStringValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStringValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private StringValueResponse _retrieveStringValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		StringValue stringValue = stringValueService.read(id);
		if (stringValue == null) throw new EntityNotFoundException("StringValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new StringValueResponse(stringValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateStringValue(@PathVariable Integer id, @Valid @RequestBody StringValueRequest stringValueRequest)
			throws DatabaseException
	{
		_updateStringValue(id, stringValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<StringValueResponse> updateStringValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute StringValueRequest stringValueRequest) throws DatabaseException
	{
		return _createStringValue(stringValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateStringValuePost(@PathVariable Integer id, @Valid @RequestBody StringValueRequest stringValueRequest)
			throws DatabaseException
	{
		_updateStringValue(id, stringValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateStringValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute StringValueRequest stringValueRequest)
			throws DatabaseException
	{
		_updateStringValue(id, stringValueRequest);
	}

	private void _updateStringValue(Integer id, StringValueRequest stringValueRequest) throws DatabaseException
	{
		StringValue stringValue = stringValueRequest.toStringValue();
		stringValue.setId(id);
		stringValueService.update(stringValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteStringValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteStringValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteStringValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteStringValue(id);
	}

	private void _deleteStringValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = stringValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("StringValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<StringValueResponse> retrieveStringValueCollection(@Valid EntityCollectionRequest stringValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStringValueCollection(stringValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<StringValueResponse> retrieveStringValueCollectionJson(@Valid EntityCollectionRequest stringValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStringValueCollection(stringValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<StringValueResponse> retrieveStringValueCollectionPost(@Valid @RequestBody EntityCollectionRequest stringValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStringValueCollection(stringValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<StringValueResponse> retrieveStringValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest stringValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveStringValueCollection(stringValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<StringValueResponse> _retrieveStringValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<StringValue> stringValuePager = stringValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<StringValueResponse>(stringValuePager, Lists.newArrayList(Iterables.transform(
				stringValuePager.getIterable(), new Function<StringValue, StringValueResponse>()
				{
					@Override
					@Nullable
					public StringValueResponse apply(@Nullable StringValue stringValue)
					{
						try
						{
							return stringValue != null ? new StringValueResponse(stringValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/stringvalue");
	}

	private static class StringValueRequest
	{
		private String value;
	
		public StringValue toStringValue()
		{
			StringValue stringValue = new StringValue();
			stringValue.setValue(value);
			return stringValue;
		}
		
		public void setValue(String value)
		{
			this.value = value;
		}
		
	}

	static class StringValueResponse
	{
		private final String href;
		private final String value;
	
		public StringValueResponse(StringValue stringValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/stringvalue/" + stringValue.getId();
			this.value = stringValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public String getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}