package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.TextValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.TextValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/textvalue")
public class TextValueController
{
	@Autowired
	private TextValueService textValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<TextValueResponse> createTextValue(@Valid @RequestBody TextValueRequest textValueRequest)
			throws DatabaseException
	{
		return _createTextValue(textValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<TextValueResponse> createTextValueFromForm(@Valid @ModelAttribute TextValueRequest textValueRequest)
			throws DatabaseException
	{
		return _createTextValue(textValueRequest);
	}

	private ResponseEntity<TextValueResponse> _createTextValue(TextValueRequest textValueRequest) throws DatabaseException
	{
		TextValue textValue = textValueService.create(textValueRequest.toTextValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/textvalue/" + textValue.getId());
		return new ResponseEntity<TextValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public TextValueResponse retrieveTextValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTextValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public TextValueResponse retrieveTextValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTextValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private TextValueResponse _retrieveTextValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		TextValue textValue = textValueService.read(id);
		if (textValue == null) throw new EntityNotFoundException("TextValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new TextValueResponse(textValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateTextValue(@PathVariable Integer id, @Valid @RequestBody TextValueRequest textValueRequest)
			throws DatabaseException
	{
		_updateTextValue(id, textValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<TextValueResponse> updateTextValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute TextValueRequest textValueRequest) throws DatabaseException
	{
		return _createTextValue(textValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateTextValuePost(@PathVariable Integer id, @Valid @RequestBody TextValueRequest textValueRequest)
			throws DatabaseException
	{
		_updateTextValue(id, textValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateTextValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute TextValueRequest textValueRequest)
			throws DatabaseException
	{
		_updateTextValue(id, textValueRequest);
	}

	private void _updateTextValue(Integer id, TextValueRequest textValueRequest) throws DatabaseException
	{
		TextValue textValue = textValueRequest.toTextValue();
		textValue.setId(id);
		textValueService.update(textValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteTextValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteTextValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteTextValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteTextValue(id);
	}

	private void _deleteTextValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = textValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("TextValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<TextValueResponse> retrieveTextValueCollection(@Valid EntityCollectionRequest textValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTextValueCollection(textValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<TextValueResponse> retrieveTextValueCollectionJson(@Valid EntityCollectionRequest textValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTextValueCollection(textValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<TextValueResponse> retrieveTextValueCollectionPost(@Valid @RequestBody EntityCollectionRequest textValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTextValueCollection(textValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<TextValueResponse> retrieveTextValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest textValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTextValueCollection(textValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<TextValueResponse> _retrieveTextValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<TextValue> textValuePager = textValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<TextValueResponse>(textValuePager, Lists.newArrayList(Iterables.transform(
				textValuePager.getIterable(), new Function<TextValue, TextValueResponse>()
				{
					@Override
					@Nullable
					public TextValueResponse apply(@Nullable TextValue textValue)
					{
						try
						{
							return textValue != null ? new TextValueResponse(textValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/textvalue");
	}

	private static class TextValueRequest
	{
		private String value;
	
		public TextValue toTextValue()
		{
			TextValue textValue = new TextValue();
			textValue.setValue(value);
			return textValue;
		}
		
		public void setValue(String value)
		{
			this.value = value;
		}
		
	}

	static class TextValueResponse
	{
		private final String href;
		private final String value;
	
		public TextValueResponse(TextValue textValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/textvalue/" + textValue.getId();
			this.value = textValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public String getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}