package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.molgenis.omx.observ.value.DateValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.DateValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/datevalue")
public class DateValueController
{
	@Autowired
	private DateValueService dateValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<DateValueResponse> createDateValue(@Valid @RequestBody DateValueRequest dateValueRequest)
			throws DatabaseException
	{
		return _createDateValue(dateValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DateValueResponse> createDateValueFromForm(@Valid @ModelAttribute DateValueRequest dateValueRequest)
			throws DatabaseException
	{
		return _createDateValue(dateValueRequest);
	}

	private ResponseEntity<DateValueResponse> _createDateValue(DateValueRequest dateValueRequest) throws DatabaseException
	{
		DateValue dateValue = dateValueService.create(dateValueRequest.toDateValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/datevalue/" + dateValue.getId());
		return new ResponseEntity<DateValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public DateValueResponse retrieveDateValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public DateValueResponse retrieveDateValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private DateValueResponse _retrieveDateValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		DateValue dateValue = dateValueService.read(id);
		if (dateValue == null) throw new EntityNotFoundException("DateValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new DateValueResponse(dateValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateDateValue(@PathVariable Integer id, @Valid @RequestBody DateValueRequest dateValueRequest)
			throws DatabaseException
	{
		_updateDateValue(id, dateValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DateValueResponse> updateDateValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute DateValueRequest dateValueRequest) throws DatabaseException
	{
		return _createDateValue(dateValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDateValuePost(@PathVariable Integer id, @Valid @RequestBody DateValueRequest dateValueRequest)
			throws DatabaseException
	{
		_updateDateValue(id, dateValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDateValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute DateValueRequest dateValueRequest)
			throws DatabaseException
	{
		_updateDateValue(id, dateValueRequest);
	}

	private void _updateDateValue(Integer id, DateValueRequest dateValueRequest) throws DatabaseException
	{
		DateValue dateValue = dateValueRequest.toDateValue();
		dateValue.setId(id);
		dateValueService.update(dateValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDateValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDateValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDateValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDateValue(id);
	}

	private void _deleteDateValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = dateValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("DateValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<DateValueResponse> retrieveDateValueCollection(@Valid EntityCollectionRequest dateValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateValueCollection(dateValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DateValueResponse> retrieveDateValueCollectionJson(@Valid EntityCollectionRequest dateValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateValueCollection(dateValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<DateValueResponse> retrieveDateValueCollectionPost(@Valid @RequestBody EntityCollectionRequest dateValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateValueCollection(dateValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DateValueResponse> retrieveDateValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest dateValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDateValueCollection(dateValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<DateValueResponse> _retrieveDateValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<DateValue> dateValuePager = dateValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<DateValueResponse>(dateValuePager, Lists.newArrayList(Iterables.transform(
				dateValuePager.getIterable(), new Function<DateValue, DateValueResponse>()
				{
					@Override
					@Nullable
					public DateValueResponse apply(@Nullable DateValue dateValue)
					{
						try
						{
							return dateValue != null ? new DateValueResponse(dateValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/datevalue");
	}

	private static class DateValueRequest
	{
		private java.util.Date value;
	
		public DateValue toDateValue()
		{
			DateValue dateValue = new DateValue();
			dateValue.setValue(value);
			return dateValue;
		}
		
		public void setValue(java.util.Date value)
		{
			this.value = value;
		}
		
	}

	static class DateValueResponse
	{
		private final String href;
		private final java.util.Date value;
	
		public DateValueResponse(DateValue dateValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/datevalue/" + dateValue.getId();
			this.value = dateValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public java.util.Date getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	public void handleEntityNotFoundException(EntityNotFoundException e)
	{
	}
	
	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	public void handleDatabaseAccessException(DatabaseAccessException e)
	{
	}
}