package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ChromosomeService;
import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.controller.OntologyTermController.OntologyTermResponse;
import org.molgenis.service.OntologyTermService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/chromosome")
public class ChromosomeController
{
	private static Logger logger = Logger.getLogger(ChromosomeController.class);
	 
	@Autowired
	private ChromosomeService chromosomeService;

	@Autowired
	private OntologyTermService ontologyTermService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ChromosomeResponse> createChromosome(@Valid @RequestBody ChromosomeRequest chromosomeRequest)
			throws DatabaseException
	{
		return _createChromosome(chromosomeRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ChromosomeResponse> createChromosomeFromForm(@Valid @ModelAttribute ChromosomeRequest chromosomeRequest)
			throws DatabaseException
	{
		return _createChromosome(chromosomeRequest);
	}

	private ResponseEntity<ChromosomeResponse> _createChromosome(ChromosomeRequest chromosomeRequest) throws DatabaseException
	{
		Chromosome chromosome = chromosomeService.create(chromosomeRequest.toChromosome());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/chromosome/" + chromosome.getId());
		return new ResponseEntity<ChromosomeResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ChromosomeResponse retrieveChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosome(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ChromosomeResponse retrieveChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosome(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ChromosomeResponse _retrieveChromosome(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Chromosome chromosome = chromosomeService.read(id);
		if (chromosome == null) throw new EntityNotFoundException("Chromosome " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ChromosomeResponse(chromosome, expandFields);
	}
			
	@RequestMapping(value = "/{id}/species", method = RequestMethod.GET)
	public String retrieveChromosomeXrefSpecies(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosomeXrefSpecies(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/species", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveChromosomeXrefSpeciesJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosomeXrefSpecies(id, "json", expandFields);
	}
	
	private String _retrieveChromosomeXrefSpecies(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Chromosome chromosome = chromosomeService.read(id);
		if (chromosome == null) throw new EntityNotFoundException("Chromosome " + id.toString() + " not found");
		Integer ontologyTermId = chromosome.getSpecies_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateChromosome(@PathVariable Integer id, @Valid @RequestBody ChromosomeRequest chromosomeRequest)
			throws DatabaseException
	{
		_updateChromosome(id, chromosomeRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ChromosomeResponse> updateChromosomeFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ChromosomeRequest chromosomeRequest) throws DatabaseException
	{
		return _createChromosome(chromosomeRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateChromosomePost(@PathVariable Integer id, @Valid @RequestBody ChromosomeRequest chromosomeRequest)
			throws DatabaseException
	{
		_updateChromosome(id, chromosomeRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateChromosomeFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ChromosomeRequest chromosomeRequest)
			throws DatabaseException
	{
		_updateChromosome(id, chromosomeRequest);
	}

	private void _updateChromosome(Integer id, ChromosomeRequest chromosomeRequest) throws DatabaseException
	{
		Chromosome chromosome = chromosomeRequest.toChromosome();
		chromosome.setId(id);
		chromosomeService.update(chromosome);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteChromosome(@PathVariable Integer id) throws DatabaseException
	{
		_deleteChromosome(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteChromosomePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteChromosome(id);
	}

	private void _deleteChromosome(Integer id) throws DatabaseException
	{
		boolean isDeleted = chromosomeService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Chromosome " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ChromosomeResponse> retrieveChromosomeCollection(@Valid EntityCollectionRequest chromosomeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosomeCollection(chromosomeCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ChromosomeResponse> retrieveChromosomeCollectionJson(@Valid EntityCollectionRequest chromosomeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosomeCollection(chromosomeCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ChromosomeResponse> retrieveChromosomeCollectionPost(@Valid @RequestBody EntityCollectionRequest chromosomeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosomeCollection(chromosomeCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ChromosomeResponse> retrieveChromosomeCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest chromosomeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveChromosomeCollection(chromosomeCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ChromosomeResponse> _retrieveChromosomeCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Chromosome> chromosomePager = chromosomeService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ChromosomeResponse>(chromosomePager, Lists.newArrayList(Iterables.transform(
				chromosomePager.getIterable(), new Function<Chromosome, ChromosomeResponse>()
				{
					@Override
					@Nullable
					public ChromosomeResponse apply(@Nullable Chromosome chromosome)
					{
						try
						{
							return chromosome != null ? new ChromosomeResponse(chromosome, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/chromosome");
	}

	private static class ChromosomeRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer orderNr;
		private Boolean isAutosomal = false;	
		private Integer bpLength;
		private Integer species;
	
		public Chromosome toChromosome()
		{
			Chromosome chromosome = new Chromosome();
			chromosome.setIdentifier(identifier);
			chromosome.setName(name);
			chromosome.setDescription(description);
			chromosome.setOrderNr(orderNr);
			chromosome.setIsAutosomal(isAutosomal);
			chromosome.setBpLength(bpLength);
			chromosome.setSpecies_Id(species);
			return chromosome;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setOrderNr(Integer orderNr)
		{
			this.orderNr = orderNr;
		}
		
		public void setIsAutosomal(Boolean isAutosomal)
		{
			this.isAutosomal = isAutosomal;
		}
		
		public void setBpLength(Integer bpLength)
		{
			this.bpLength = bpLength;
		}
		
		public void setSpecies(Integer species)
		{
			this.species = species;
		}
		
	}

	static class ChromosomeResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Integer orderNr;
		private final Boolean isAutosomal;
		private final Integer bpLength;
		private final Object species;
	
		public ChromosomeResponse(Chromosome chromosome, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/chromosome/" + chromosome.getId();
			this.identifier = chromosome.getIdentifier();
			this.name = chromosome.getName();
			this.description = chromosome.getDescription();
			this.orderNr = chromosome.getOrderNr();
			this.isAutosomal = chromosome.getIsAutosomal();
			this.bpLength = chromosome.getBpLength();
			if (expandFields != null && expandFields.contains("species")) this.species = chromosome.getSpecies() == null ? null : new OntologyTermResponse(chromosome.getSpecies(), null);
			else this.species = chromosome.getSpecies() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/chromosome/" + chromosome.getId() + "/species");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Integer getOrderNr()
		{
			return orderNr;
		}
	
		public Boolean getIsAutosomal()
		{
			return isAutosomal;
		}
	
		public Integer getBpLength()
		{
			return bpLength;
		}
	
		public Object getSpecies()
		{
			return species;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}