package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.DataSet;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.DataSetService;
import org.molgenis.omx.observ.Protocol;
import org.molgenis.controller.ProtocolController.ProtocolResponse;
import org.molgenis.service.ProtocolService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/dataset")
public class DataSetController
{
	private static Logger logger = Logger.getLogger(DataSetController.class);
	 
	@Autowired
	private DataSetService dataSetService;

	@Autowired
	private ProtocolService protocolService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<DataSetResponse> createDataSet(@Valid @RequestBody DataSetRequest dataSetRequest)
			throws DatabaseException
	{
		return _createDataSet(dataSetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DataSetResponse> createDataSetFromForm(@Valid @ModelAttribute DataSetRequest dataSetRequest)
			throws DatabaseException
	{
		return _createDataSet(dataSetRequest);
	}

	private ResponseEntity<DataSetResponse> _createDataSet(DataSetRequest dataSetRequest) throws DatabaseException
	{
		DataSet dataSet = dataSetService.create(dataSetRequest.toDataSet());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/dataset/" + dataSet.getId());
		return new ResponseEntity<DataSetResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public DataSetResponse retrieveDataSet(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSet(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public DataSetResponse retrieveDataSetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSet(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private DataSetResponse _retrieveDataSet(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		DataSet dataSet = dataSetService.read(id);
		if (dataSet == null) throw new EntityNotFoundException("DataSet " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new DataSetResponse(dataSet, expandFields);
	}
			
	@RequestMapping(value = "/{id}/protocolUsed", method = RequestMethod.GET)
	public String retrieveDataSetXrefProtocolUsed(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSetXrefProtocolUsed(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/protocolUsed", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveDataSetXrefProtocolUsedJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSetXrefProtocolUsed(id, "json", expandFields);
	}
	
	private String _retrieveDataSetXrefProtocolUsed(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		DataSet dataSet = dataSetService.read(id);
		if (dataSet == null) throw new EntityNotFoundException("DataSet " + id.toString() + " not found");
		Integer protocolId = dataSet.getProtocolUsed_Id();
		String redirectUri = "redirect:/api/v1/protocol/" + protocolId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateDataSet(@PathVariable Integer id, @Valid @RequestBody DataSetRequest dataSetRequest)
			throws DatabaseException
	{
		_updateDataSet(id, dataSetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DataSetResponse> updateDataSetFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute DataSetRequest dataSetRequest) throws DatabaseException
	{
		return _createDataSet(dataSetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDataSetPost(@PathVariable Integer id, @Valid @RequestBody DataSetRequest dataSetRequest)
			throws DatabaseException
	{
		_updateDataSet(id, dataSetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDataSetFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute DataSetRequest dataSetRequest)
			throws DatabaseException
	{
		_updateDataSet(id, dataSetRequest);
	}

	private void _updateDataSet(Integer id, DataSetRequest dataSetRequest) throws DatabaseException
	{
		DataSet dataSet = dataSetRequest.toDataSet();
		dataSet.setId(id);
		dataSetService.update(dataSet);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDataSet(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDataSet(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDataSetPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDataSet(id);
	}

	private void _deleteDataSet(Integer id) throws DatabaseException
	{
		boolean isDeleted = dataSetService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("DataSet " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<DataSetResponse> retrieveDataSetCollection(@Valid EntityCollectionRequest dataSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSetCollection(dataSetCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DataSetResponse> retrieveDataSetCollectionJson(@Valid EntityCollectionRequest dataSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSetCollection(dataSetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<DataSetResponse> retrieveDataSetCollectionPost(@Valid @RequestBody EntityCollectionRequest dataSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSetCollection(dataSetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DataSetResponse> retrieveDataSetCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest dataSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDataSetCollection(dataSetCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<DataSetResponse> _retrieveDataSetCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<DataSet> dataSetPager = dataSetService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<DataSetResponse>(dataSetPager, Lists.newArrayList(Iterables.transform(
				dataSetPager.getIterable(), new Function<DataSet, DataSetResponse>()
				{
					@Override
					@Nullable
					public DataSetResponse apply(@Nullable DataSet dataSet)
					{
						try
						{
							return dataSet != null ? new DataSetResponse(dataSet, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/dataset");
	}

	private static class DataSetRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer protocolUsed;
		private java.util.Date startTime;
		private java.util.Date endTime;
		private Boolean active = false;	
	
		public DataSet toDataSet()
		{
			DataSet dataSet = new DataSet();
			dataSet.setIdentifier(identifier);
			dataSet.setName(name);
			dataSet.setDescription(description);
			dataSet.setProtocolUsed_Id(protocolUsed);
			dataSet.setStartTime(startTime);
			dataSet.setEndTime(endTime);
			dataSet.setActive(active);
			return dataSet;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setProtocolUsed(Integer protocolUsed)
		{
			this.protocolUsed = protocolUsed;
		}
		
		public void setStartTime(java.util.Date startTime)
		{
			this.startTime = startTime;
		}
		
		public void setEndTime(java.util.Date endTime)
		{
			this.endTime = endTime;
		}
		
		public void setActive(Boolean active)
		{
			this.active = active;
		}
		
	}

	static class DataSetResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object protocolUsed;
		private final java.util.Date startTime;
		private final java.util.Date endTime;
		private final Boolean active;
	
		public DataSetResponse(DataSet dataSet, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/dataset/" + dataSet.getId();
			this.identifier = dataSet.getIdentifier();
			this.name = dataSet.getName();
			this.description = dataSet.getDescription();
			if (expandFields != null && expandFields.contains("protocolUsed")) this.protocolUsed = new ProtocolResponse(dataSet.getProtocolUsed(), null);
			else this.protocolUsed = java.util.Collections.singletonMap("href", "/api/v1/dataset/" + dataSet.getId() + "/protocolUsed");
			this.startTime = dataSet.getStartTime();
			this.endTime = dataSet.getEndTime();
			this.active = dataSet.getActive();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getProtocolUsed()
		{
			return protocolUsed;
		}
	
		public java.util.Date getStartTime()
		{
			return startTime;
		}
	
		public java.util.Date getEndTime()
		{
			return endTime;
		}
	
		public Boolean getActive()
		{
			return active;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}