package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.DerivedTrait;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.DerivedTraitService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/derivedtrait")
public class DerivedTraitController
{
	private static Logger logger = Logger.getLogger(DerivedTraitController.class);
	 
	@Autowired
	private DerivedTraitService derivedTraitService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<DerivedTraitResponse> createDerivedTrait(@Valid @RequestBody DerivedTraitRequest derivedTraitRequest)
			throws DatabaseException
	{
		return _createDerivedTrait(derivedTraitRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DerivedTraitResponse> createDerivedTraitFromForm(@Valid @ModelAttribute DerivedTraitRequest derivedTraitRequest)
			throws DatabaseException
	{
		return _createDerivedTrait(derivedTraitRequest);
	}

	private ResponseEntity<DerivedTraitResponse> _createDerivedTrait(DerivedTraitRequest derivedTraitRequest) throws DatabaseException
	{
		DerivedTrait derivedTrait = derivedTraitService.create(derivedTraitRequest.toDerivedTrait());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/derivedtrait/" + derivedTrait.getId());
		return new ResponseEntity<DerivedTraitResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public DerivedTraitResponse retrieveDerivedTrait(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDerivedTrait(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public DerivedTraitResponse retrieveDerivedTraitJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDerivedTrait(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private DerivedTraitResponse _retrieveDerivedTrait(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		DerivedTrait derivedTrait = derivedTraitService.read(id);
		if (derivedTrait == null) throw new EntityNotFoundException("DerivedTrait " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new DerivedTraitResponse(derivedTrait, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateDerivedTrait(@PathVariable Integer id, @Valid @RequestBody DerivedTraitRequest derivedTraitRequest)
			throws DatabaseException
	{
		_updateDerivedTrait(id, derivedTraitRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DerivedTraitResponse> updateDerivedTraitFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute DerivedTraitRequest derivedTraitRequest) throws DatabaseException
	{
		return _createDerivedTrait(derivedTraitRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDerivedTraitPost(@PathVariable Integer id, @Valid @RequestBody DerivedTraitRequest derivedTraitRequest)
			throws DatabaseException
	{
		_updateDerivedTrait(id, derivedTraitRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDerivedTraitFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute DerivedTraitRequest derivedTraitRequest)
			throws DatabaseException
	{
		_updateDerivedTrait(id, derivedTraitRequest);
	}

	private void _updateDerivedTrait(Integer id, DerivedTraitRequest derivedTraitRequest) throws DatabaseException
	{
		DerivedTrait derivedTrait = derivedTraitRequest.toDerivedTrait();
		derivedTrait.setId(id);
		derivedTraitService.update(derivedTrait);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDerivedTrait(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDerivedTrait(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDerivedTraitPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDerivedTrait(id);
	}

	private void _deleteDerivedTrait(Integer id) throws DatabaseException
	{
		boolean isDeleted = derivedTraitService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("DerivedTrait " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<DerivedTraitResponse> retrieveDerivedTraitCollection(@Valid EntityCollectionRequest derivedTraitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDerivedTraitCollection(derivedTraitCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DerivedTraitResponse> retrieveDerivedTraitCollectionJson(@Valid EntityCollectionRequest derivedTraitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDerivedTraitCollection(derivedTraitCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<DerivedTraitResponse> retrieveDerivedTraitCollectionPost(@Valid @RequestBody EntityCollectionRequest derivedTraitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDerivedTraitCollection(derivedTraitCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DerivedTraitResponse> retrieveDerivedTraitCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest derivedTraitCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDerivedTraitCollection(derivedTraitCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<DerivedTraitResponse> _retrieveDerivedTraitCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<DerivedTrait> derivedTraitPager = derivedTraitService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<DerivedTraitResponse>(derivedTraitPager, Lists.newArrayList(Iterables.transform(
				derivedTraitPager.getIterable(), new Function<DerivedTrait, DerivedTraitResponse>()
				{
					@Override
					@Nullable
					public DerivedTraitResponse apply(@Nullable DerivedTrait derivedTrait)
					{
						try
						{
							return derivedTrait != null ? new DerivedTraitResponse(derivedTrait, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/derivedtrait");
	}

	private static class DerivedTraitRequest
	{
		private String identifier;
		private String name;
		private String description;
	
		public DerivedTrait toDerivedTrait()
		{
			DerivedTrait derivedTrait = new DerivedTrait();
			derivedTrait.setIdentifier(identifier);
			derivedTrait.setName(name);
			derivedTrait.setDescription(description);
			return derivedTrait;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
	}

	static class DerivedTraitResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
	
		public DerivedTraitResponse(DerivedTrait derivedTrait, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/derivedtrait/" + derivedTrait.getId();
			this.identifier = derivedTrait.getIdentifier();
			this.name = derivedTrait.getName();
			this.description = derivedTrait.getDescription();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}