package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Gene;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.GeneService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/gene")
public class GeneController
{
	private static Logger logger = Logger.getLogger(GeneController.class);
	 
	@Autowired
	private GeneService geneService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<GeneResponse> createGene(@Valid @RequestBody GeneRequest geneRequest)
			throws DatabaseException
	{
		return _createGene(geneRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<GeneResponse> createGeneFromForm(@Valid @ModelAttribute GeneRequest geneRequest)
			throws DatabaseException
	{
		return _createGene(geneRequest);
	}

	private ResponseEntity<GeneResponse> _createGene(GeneRequest geneRequest) throws DatabaseException
	{
		Gene gene = geneService.create(geneRequest.toGene());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/gene/" + gene.getId());
		return new ResponseEntity<GeneResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public GeneResponse retrieveGene(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGene(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public GeneResponse retrieveGeneJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGene(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private GeneResponse _retrieveGene(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Gene gene = geneService.read(id);
		if (gene == null) throw new EntityNotFoundException("Gene " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new GeneResponse(gene, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveGeneXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGeneXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveGeneXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGeneXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveGeneXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Gene gene = geneService.read(id);
		if (gene == null) throw new EntityNotFoundException("Gene " + id.toString() + " not found");
		Integer chromosomeId = gene.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateGene(@PathVariable Integer id, @Valid @RequestBody GeneRequest geneRequest)
			throws DatabaseException
	{
		_updateGene(id, geneRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<GeneResponse> updateGeneFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute GeneRequest geneRequest) throws DatabaseException
	{
		return _createGene(geneRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateGenePost(@PathVariable Integer id, @Valid @RequestBody GeneRequest geneRequest)
			throws DatabaseException
	{
		_updateGene(id, geneRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateGeneFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute GeneRequest geneRequest)
			throws DatabaseException
	{
		_updateGene(id, geneRequest);
	}

	private void _updateGene(Integer id, GeneRequest geneRequest) throws DatabaseException
	{
		Gene gene = geneRequest.toGene();
		gene.setId(id);
		geneService.update(gene);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteGene(@PathVariable Integer id) throws DatabaseException
	{
		_deleteGene(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteGenePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteGene(id);
	}

	private void _deleteGene(Integer id) throws DatabaseException
	{
		boolean isDeleted = geneService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Gene " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<GeneResponse> retrieveGeneCollection(@Valid EntityCollectionRequest geneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGeneCollection(geneCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<GeneResponse> retrieveGeneCollectionJson(@Valid EntityCollectionRequest geneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGeneCollection(geneCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<GeneResponse> retrieveGeneCollectionPost(@Valid @RequestBody EntityCollectionRequest geneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGeneCollection(geneCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<GeneResponse> retrieveGeneCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest geneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGeneCollection(geneCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<GeneResponse> _retrieveGeneCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Gene> genePager = geneService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<GeneResponse>(genePager, Lists.newArrayList(Iterables.transform(
				genePager.getIterable(), new Function<Gene, GeneResponse>()
				{
					@Override
					@Nullable
					public GeneResponse apply(@Nullable Gene gene)
					{
						try
						{
							return gene != null ? new GeneResponse(gene, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/gene");
	}

	private static class GeneRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
		private String orientation;
		private Boolean control = false;	
	
		public Gene toGene()
		{
			Gene gene = new Gene();
			gene.setIdentifier(identifier);
			gene.setName(name);
			gene.setDescription(description);
			gene.setChromosome_Id(chromosome);
			gene.setCM(cM);
			gene.setBpStart(bpStart);
			gene.setBpEnd(bpEnd);
			gene.setSeq(seq);
			gene.setSymbol(symbol);
			gene.setOrientation(orientation);
			gene.setControl(control);
			return gene;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
		public void setOrientation(String orientation)
		{
			this.orientation = orientation;
		}
		
		public void setControl(Boolean control)
		{
			this.control = control;
		}
		
	}

	static class GeneResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
		private final String orientation;
		private final Boolean control;
	
		public GeneResponse(Gene gene, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/gene/" + gene.getId();
			this.identifier = gene.getIdentifier();
			this.name = gene.getName();
			this.description = gene.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = gene.getChromosome() == null ? null : new ChromosomeResponse(gene.getChromosome(), null);
			else this.chromosome = gene.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/gene/" + gene.getId() + "/chromosome");
			this.cM = gene.getCM();
			this.bpStart = gene.getBpStart();
			this.bpEnd = gene.getBpEnd();
			this.seq = gene.getSeq();
			this.symbol = gene.getSymbol();
			this.orientation = gene.getOrientation();
			this.control = gene.getControl();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
		public String getOrientation()
		{
			return orientation;
		}
	
		public Boolean getControl()
		{
			return control;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}