package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.auth.GroupAuthority;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.GroupAuthorityService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/groupauthority")
public class GroupAuthorityController
{
	private static Logger logger = Logger.getLogger(GroupAuthorityController.class);
	 
	@Autowired
	private GroupAuthorityService groupAuthorityService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<GroupAuthorityResponse> createGroupAuthority(@Valid @RequestBody GroupAuthorityRequest groupAuthorityRequest)
			throws DatabaseException
	{
		return _createGroupAuthority(groupAuthorityRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<GroupAuthorityResponse> createGroupAuthorityFromForm(@Valid @ModelAttribute GroupAuthorityRequest groupAuthorityRequest)
			throws DatabaseException
	{
		return _createGroupAuthority(groupAuthorityRequest);
	}

	private ResponseEntity<GroupAuthorityResponse> _createGroupAuthority(GroupAuthorityRequest groupAuthorityRequest) throws DatabaseException
	{
		GroupAuthority groupAuthority = groupAuthorityService.create(groupAuthorityRequest.toGroupAuthority());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/groupauthority/" + groupAuthority.getId());
		return new ResponseEntity<GroupAuthorityResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public GroupAuthorityResponse retrieveGroupAuthority(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthority(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public GroupAuthorityResponse retrieveGroupAuthorityJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthority(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private GroupAuthorityResponse _retrieveGroupAuthority(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		GroupAuthority groupAuthority = groupAuthorityService.read(id);
		if (groupAuthority == null) throw new EntityNotFoundException("GroupAuthority " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new GroupAuthorityResponse(groupAuthority, expandFields);
	}
			
	@RequestMapping(value = "/{id}/molgenisGroup", method = RequestMethod.GET)
	public String retrieveGroupAuthorityXrefMolgenisGroup(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthorityXrefMolgenisGroup(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/molgenisGroup", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveGroupAuthorityXrefMolgenisGroupJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthorityXrefMolgenisGroup(id, "json", expandFields);
	}
	
	private String _retrieveGroupAuthorityXrefMolgenisGroup(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		GroupAuthority groupAuthority = groupAuthorityService.read(id);
		if (groupAuthority == null) throw new EntityNotFoundException("GroupAuthority " + id.toString() + " not found");
		Integer molgenisGroupId = groupAuthority.getMolgenisGroup_Id();
		String redirectUri = "redirect:/api/v1/molgenisgroup/" + molgenisGroupId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateGroupAuthority(@PathVariable Integer id, @Valid @RequestBody GroupAuthorityRequest groupAuthorityRequest)
			throws DatabaseException
	{
		_updateGroupAuthority(id, groupAuthorityRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<GroupAuthorityResponse> updateGroupAuthorityFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute GroupAuthorityRequest groupAuthorityRequest) throws DatabaseException
	{
		return _createGroupAuthority(groupAuthorityRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateGroupAuthorityPost(@PathVariable Integer id, @Valid @RequestBody GroupAuthorityRequest groupAuthorityRequest)
			throws DatabaseException
	{
		_updateGroupAuthority(id, groupAuthorityRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateGroupAuthorityFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute GroupAuthorityRequest groupAuthorityRequest)
			throws DatabaseException
	{
		_updateGroupAuthority(id, groupAuthorityRequest);
	}

	private void _updateGroupAuthority(Integer id, GroupAuthorityRequest groupAuthorityRequest) throws DatabaseException
	{
		GroupAuthority groupAuthority = groupAuthorityRequest.toGroupAuthority();
		groupAuthority.setId(id);
		groupAuthorityService.update(groupAuthority);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteGroupAuthority(@PathVariable Integer id) throws DatabaseException
	{
		_deleteGroupAuthority(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteGroupAuthorityPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteGroupAuthority(id);
	}

	private void _deleteGroupAuthority(Integer id) throws DatabaseException
	{
		boolean isDeleted = groupAuthorityService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("GroupAuthority " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<GroupAuthorityResponse> retrieveGroupAuthorityCollection(@Valid EntityCollectionRequest groupAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthorityCollection(groupAuthorityCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<GroupAuthorityResponse> retrieveGroupAuthorityCollectionJson(@Valid EntityCollectionRequest groupAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthorityCollection(groupAuthorityCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<GroupAuthorityResponse> retrieveGroupAuthorityCollectionPost(@Valid @RequestBody EntityCollectionRequest groupAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthorityCollection(groupAuthorityCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<GroupAuthorityResponse> retrieveGroupAuthorityCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest groupAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveGroupAuthorityCollection(groupAuthorityCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<GroupAuthorityResponse> _retrieveGroupAuthorityCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<GroupAuthority> groupAuthorityPager = groupAuthorityService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<GroupAuthorityResponse>(groupAuthorityPager, Lists.newArrayList(Iterables.transform(
				groupAuthorityPager.getIterable(), new Function<GroupAuthority, GroupAuthorityResponse>()
				{
					@Override
					@Nullable
					public GroupAuthorityResponse apply(@Nullable GroupAuthority groupAuthority)
					{
						try
						{
							return groupAuthority != null ? new GroupAuthorityResponse(groupAuthority, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/groupauthority");
	}

	private static class GroupAuthorityRequest
	{
		private String role;
	
		public GroupAuthority toGroupAuthority()
		{
			GroupAuthority groupAuthority = new GroupAuthority();
			groupAuthority.setRole(role);
			return groupAuthority;
		}
		
		public void setRole(String role)
		{
			this.role = role;
		}
		
		
	}

	static class GroupAuthorityResponse
	{
		private final String role;
		private final String href;
	
		public GroupAuthorityResponse(GroupAuthority groupAuthority, final Set<String> expandFields) throws DatabaseException
		{
			this.role = groupAuthority.getRole();
			this.href = "/api/v1/groupauthority/" + groupAuthority.getId();
		}
	
		public String getRole()
		{
			return role;
		}
	
		public String getHref()
		{
			return href;
		}
	
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}