package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.value.LongValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.LongValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/longvalue")
public class LongValueController
{
	private static Logger logger = Logger.getLogger(LongValueController.class);
	 
	@Autowired
	private LongValueService longValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<LongValueResponse> createLongValue(@Valid @RequestBody LongValueRequest longValueRequest)
			throws DatabaseException
	{
		return _createLongValue(longValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<LongValueResponse> createLongValueFromForm(@Valid @ModelAttribute LongValueRequest longValueRequest)
			throws DatabaseException
	{
		return _createLongValue(longValueRequest);
	}

	private ResponseEntity<LongValueResponse> _createLongValue(LongValueRequest longValueRequest) throws DatabaseException
	{
		LongValue longValue = longValueService.create(longValueRequest.toLongValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/longvalue/" + longValue.getId());
		return new ResponseEntity<LongValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public LongValueResponse retrieveLongValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveLongValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public LongValueResponse retrieveLongValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveLongValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private LongValueResponse _retrieveLongValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		LongValue longValue = longValueService.read(id);
		if (longValue == null) throw new EntityNotFoundException("LongValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new LongValueResponse(longValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateLongValue(@PathVariable Integer id, @Valid @RequestBody LongValueRequest longValueRequest)
			throws DatabaseException
	{
		_updateLongValue(id, longValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<LongValueResponse> updateLongValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute LongValueRequest longValueRequest) throws DatabaseException
	{
		return _createLongValue(longValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateLongValuePost(@PathVariable Integer id, @Valid @RequestBody LongValueRequest longValueRequest)
			throws DatabaseException
	{
		_updateLongValue(id, longValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateLongValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute LongValueRequest longValueRequest)
			throws DatabaseException
	{
		_updateLongValue(id, longValueRequest);
	}

	private void _updateLongValue(Integer id, LongValueRequest longValueRequest) throws DatabaseException
	{
		LongValue longValue = longValueRequest.toLongValue();
		longValue.setId(id);
		longValueService.update(longValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteLongValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteLongValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteLongValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteLongValue(id);
	}

	private void _deleteLongValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = longValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("LongValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<LongValueResponse> retrieveLongValueCollection(@Valid EntityCollectionRequest longValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveLongValueCollection(longValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<LongValueResponse> retrieveLongValueCollectionJson(@Valid EntityCollectionRequest longValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveLongValueCollection(longValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<LongValueResponse> retrieveLongValueCollectionPost(@Valid @RequestBody EntityCollectionRequest longValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveLongValueCollection(longValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<LongValueResponse> retrieveLongValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest longValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveLongValueCollection(longValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<LongValueResponse> _retrieveLongValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<LongValue> longValuePager = longValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<LongValueResponse>(longValuePager, Lists.newArrayList(Iterables.transform(
				longValuePager.getIterable(), new Function<LongValue, LongValueResponse>()
				{
					@Override
					@Nullable
					public LongValueResponse apply(@Nullable LongValue longValue)
					{
						try
						{
							return longValue != null ? new LongValueResponse(longValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/longvalue");
	}

	private static class LongValueRequest
	{
		private Long value;
	
		public LongValue toLongValue()
		{
			LongValue longValue = new LongValue();
			longValue.setValue(value);
			return longValue;
		}
		
		public void setValue(Long value)
		{
			this.value = value;
		}
		
	}

	static class LongValueResponse
	{
		private final String href;
		private final Long value;
	
		public LongValueResponse(LongValue longValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/longvalue/" + longValue.getId();
			this.value = longValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public Long getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}