package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.NMRBin;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.NMRBinService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/nmrbin")
public class NMRBinController
{
	private static Logger logger = Logger.getLogger(NMRBinController.class);
	 
	@Autowired
	private NMRBinService nMRBinService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<NMRBinResponse> createNMRBin(@Valid @RequestBody NMRBinRequest nMRBinRequest)
			throws DatabaseException
	{
		return _createNMRBin(nMRBinRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<NMRBinResponse> createNMRBinFromForm(@Valid @ModelAttribute NMRBinRequest nMRBinRequest)
			throws DatabaseException
	{
		return _createNMRBin(nMRBinRequest);
	}

	private ResponseEntity<NMRBinResponse> _createNMRBin(NMRBinRequest nMRBinRequest) throws DatabaseException
	{
		NMRBin nMRBin = nMRBinService.create(nMRBinRequest.toNMRBin());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/nmrbin/" + nMRBin.getId());
		return new ResponseEntity<NMRBinResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public NMRBinResponse retrieveNMRBin(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNMRBin(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public NMRBinResponse retrieveNMRBinJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNMRBin(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private NMRBinResponse _retrieveNMRBin(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		NMRBin nMRBin = nMRBinService.read(id);
		if (nMRBin == null) throw new EntityNotFoundException("NMRBin " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new NMRBinResponse(nMRBin, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateNMRBin(@PathVariable Integer id, @Valid @RequestBody NMRBinRequest nMRBinRequest)
			throws DatabaseException
	{
		_updateNMRBin(id, nMRBinRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<NMRBinResponse> updateNMRBinFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute NMRBinRequest nMRBinRequest) throws DatabaseException
	{
		return _createNMRBin(nMRBinRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateNMRBinPost(@PathVariable Integer id, @Valid @RequestBody NMRBinRequest nMRBinRequest)
			throws DatabaseException
	{
		_updateNMRBin(id, nMRBinRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateNMRBinFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute NMRBinRequest nMRBinRequest)
			throws DatabaseException
	{
		_updateNMRBin(id, nMRBinRequest);
	}

	private void _updateNMRBin(Integer id, NMRBinRequest nMRBinRequest) throws DatabaseException
	{
		NMRBin nMRBin = nMRBinRequest.toNMRBin();
		nMRBin.setId(id);
		nMRBinService.update(nMRBin);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteNMRBin(@PathVariable Integer id) throws DatabaseException
	{
		_deleteNMRBin(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteNMRBinPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteNMRBin(id);
	}

	private void _deleteNMRBin(Integer id) throws DatabaseException
	{
		boolean isDeleted = nMRBinService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("NMRBin " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<NMRBinResponse> retrieveNMRBinCollection(@Valid EntityCollectionRequest nMRBinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNMRBinCollection(nMRBinCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<NMRBinResponse> retrieveNMRBinCollectionJson(@Valid EntityCollectionRequest nMRBinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNMRBinCollection(nMRBinCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<NMRBinResponse> retrieveNMRBinCollectionPost(@Valid @RequestBody EntityCollectionRequest nMRBinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNMRBinCollection(nMRBinCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<NMRBinResponse> retrieveNMRBinCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest nMRBinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveNMRBinCollection(nMRBinCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<NMRBinResponse> _retrieveNMRBinCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<NMRBin> nMRBinPager = nMRBinService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<NMRBinResponse>(nMRBinPager, Lists.newArrayList(Iterables.transform(
				nMRBinPager.getIterable(), new Function<NMRBin, NMRBinResponse>()
				{
					@Override
					@Nullable
					public NMRBinResponse apply(@Nullable NMRBin nMRBin)
					{
						try
						{
							return nMRBin != null ? new NMRBinResponse(nMRBin, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/nmrbin");
	}

	private static class NMRBinRequest
	{
		private String identifier;
		private String name;
		private String description;
	
		public NMRBin toNMRBin()
		{
			NMRBin nMRBin = new NMRBin();
			nMRBin.setIdentifier(identifier);
			nMRBin.setName(name);
			nMRBin.setDescription(description);
			return nMRBin;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
	}

	static class NMRBinResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
	
		public NMRBinResponse(NMRBin nMRBin, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/nmrbin/" + nMRBin.getId();
			this.identifier = nMRBin.getIdentifier();
			this.name = nMRBin.getName();
			this.description = nMRBin.getDescription();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}