package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Variant;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.VariantService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.omx.xgap.Track;
import org.molgenis.controller.TrackController.TrackResponse;
import org.molgenis.service.TrackService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/variant")
public class VariantController
{
	private static Logger logger = Logger.getLogger(VariantController.class);
	 
	@Autowired
	private VariantService variantService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@Autowired
	private TrackService trackService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<VariantResponse> createVariant(@Valid @RequestBody VariantRequest variantRequest)
			throws DatabaseException
	{
		return _createVariant(variantRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<VariantResponse> createVariantFromForm(@Valid @ModelAttribute VariantRequest variantRequest)
			throws DatabaseException
	{
		return _createVariant(variantRequest);
	}

	private ResponseEntity<VariantResponse> _createVariant(VariantRequest variantRequest) throws DatabaseException
	{
		Variant variant = variantService.create(variantRequest.toVariant());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/variant/" + variant.getId());
		return new ResponseEntity<VariantResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public VariantResponse retrieveVariant(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariant(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public VariantResponse retrieveVariantJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariant(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private VariantResponse _retrieveVariant(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Variant variant = variantService.read(id);
		if (variant == null) throw new EntityNotFoundException("Variant " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new VariantResponse(variant, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveVariantXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveVariantXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveVariantXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Variant variant = variantService.read(id);
		if (variant == null) throw new EntityNotFoundException("Variant " + id.toString() + " not found");
		Integer chromosomeId = variant.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/track", method = RequestMethod.GET)
	public String retrieveVariantXrefTrack(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantXrefTrack(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/track", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveVariantXrefTrackJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantXrefTrack(id, "json", expandFields);
	}
	
	private String _retrieveVariantXrefTrack(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Variant variant = variantService.read(id);
		if (variant == null) throw new EntityNotFoundException("Variant " + id.toString() + " not found");
		Integer trackId = variant.getTrack_Id();
		String redirectUri = "redirect:/api/v1/track/" + trackId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateVariant(@PathVariable Integer id, @Valid @RequestBody VariantRequest variantRequest)
			throws DatabaseException
	{
		_updateVariant(id, variantRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<VariantResponse> updateVariantFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute VariantRequest variantRequest) throws DatabaseException
	{
		return _createVariant(variantRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateVariantPost(@PathVariable Integer id, @Valid @RequestBody VariantRequest variantRequest)
			throws DatabaseException
	{
		_updateVariant(id, variantRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateVariantFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute VariantRequest variantRequest)
			throws DatabaseException
	{
		_updateVariant(id, variantRequest);
	}

	private void _updateVariant(Integer id, VariantRequest variantRequest) throws DatabaseException
	{
		Variant variant = variantRequest.toVariant();
		variant.setId(id);
		variantService.update(variant);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteVariant(@PathVariable Integer id) throws DatabaseException
	{
		_deleteVariant(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteVariantPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteVariant(id);
	}

	private void _deleteVariant(Integer id) throws DatabaseException
	{
		boolean isDeleted = variantService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Variant " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<VariantResponse> retrieveVariantCollection(@Valid EntityCollectionRequest variantCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantCollection(variantCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<VariantResponse> retrieveVariantCollectionJson(@Valid EntityCollectionRequest variantCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantCollection(variantCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<VariantResponse> retrieveVariantCollectionPost(@Valid @RequestBody EntityCollectionRequest variantCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantCollection(variantCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<VariantResponse> retrieveVariantCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest variantCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveVariantCollection(variantCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<VariantResponse> _retrieveVariantCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Variant> variantPager = variantService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<VariantResponse>(variantPager, Lists.newArrayList(Iterables.transform(
				variantPager.getIterable(), new Function<Variant, VariantResponse>()
				{
					@Override
					@Nullable
					public VariantResponse apply(@Nullable Variant variant)
					{
						try
						{
							return variant != null ? new VariantResponse(variant, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/variant");
	}

	private static class VariantRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
		private Integer mutationPosition;
		private Integer cdnaPosition;
		private Integer aaPosition;
		private Integer variantLength;
		private String event;
		private String ntChange;
		private String codonchange;
		private String cdnaNotation;
		private String gdnaNotation;
		private String aaNotation;
		private String exon;
		private String consequence;
		private String inheritance;
		private Boolean reportedSNP = false;	
		private Boolean effectOnSplicing = false;	
		private String pathogenicity;
		private String gene;
		private Integer idMutation;
		private String detailsForMutation;
		private Integer track;
	
		public Variant toVariant()
		{
			Variant variant = new Variant();
			variant.setIdentifier(identifier);
			variant.setName(name);
			variant.setDescription(description);
			variant.setChromosome_Id(chromosome);
			variant.setCM(cM);
			variant.setBpStart(bpStart);
			variant.setBpEnd(bpEnd);
			variant.setSeq(seq);
			variant.setSymbol(symbol);
			variant.setMutationPosition(mutationPosition);
			variant.setCdnaPosition(cdnaPosition);
			variant.setAaPosition(aaPosition);
			variant.setVariantLength(variantLength);
			variant.setEvent(event);
			variant.setNtChange(ntChange);
			variant.setCodonchange(codonchange);
			variant.setCdnaNotation(cdnaNotation);
			variant.setGdnaNotation(gdnaNotation);
			variant.setAaNotation(aaNotation);
			variant.setExon(exon);
			variant.setConsequence(consequence);
			variant.setInheritance(inheritance);
			variant.setReportedSNP(reportedSNP);
			variant.setEffectOnSplicing(effectOnSplicing);
			variant.setPathogenicity(pathogenicity);
			variant.setGene(gene);
			variant.setIdMutation(idMutation);
			variant.setDetailsForMutation(detailsForMutation);
			variant.setTrack_Id(track);
			return variant;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
		public void setMutationPosition(Integer mutationPosition)
		{
			this.mutationPosition = mutationPosition;
		}
		
		public void setCdnaPosition(Integer cdnaPosition)
		{
			this.cdnaPosition = cdnaPosition;
		}
		
		public void setAaPosition(Integer aaPosition)
		{
			this.aaPosition = aaPosition;
		}
		
		public void setVariantLength(Integer variantLength)
		{
			this.variantLength = variantLength;
		}
		
		public void setEvent(String event)
		{
			this.event = event;
		}
		
		public void setNtChange(String ntChange)
		{
			this.ntChange = ntChange;
		}
		
		public void setCodonchange(String codonchange)
		{
			this.codonchange = codonchange;
		}
		
		public void setCdnaNotation(String cdnaNotation)
		{
			this.cdnaNotation = cdnaNotation;
		}
		
		public void setGdnaNotation(String gdnaNotation)
		{
			this.gdnaNotation = gdnaNotation;
		}
		
		public void setAaNotation(String aaNotation)
		{
			this.aaNotation = aaNotation;
		}
		
		public void setExon(String exon)
		{
			this.exon = exon;
		}
		
		public void setConsequence(String consequence)
		{
			this.consequence = consequence;
		}
		
		public void setInheritance(String inheritance)
		{
			this.inheritance = inheritance;
		}
		
		public void setReportedSNP(Boolean reportedSNP)
		{
			this.reportedSNP = reportedSNP;
		}
		
		public void setEffectOnSplicing(Boolean effectOnSplicing)
		{
			this.effectOnSplicing = effectOnSplicing;
		}
		
		public void setPathogenicity(String pathogenicity)
		{
			this.pathogenicity = pathogenicity;
		}
		
		public void setGene(String gene)
		{
			this.gene = gene;
		}
		
		public void setIdMutation(Integer idMutation)
		{
			this.idMutation = idMutation;
		}
		
		public void setDetailsForMutation(String detailsForMutation)
		{
			this.detailsForMutation = detailsForMutation;
		}
		
		public void setTrack(Integer track)
		{
			this.track = track;
		}
		
	}

	static class VariantResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
		private final Integer mutationPosition;
		private final Integer cdnaPosition;
		private final Integer aaPosition;
		private final Integer variantLength;
		private final String event;
		private final String ntChange;
		private final String codonchange;
		private final String cdnaNotation;
		private final String gdnaNotation;
		private final String aaNotation;
		private final String exon;
		private final String consequence;
		private final String inheritance;
		private final Boolean reportedSNP;
		private final Boolean effectOnSplicing;
		private final String pathogenicity;
		private final String gene;
		private final Integer idMutation;
		private final String detailsForMutation;
		private final Object track;
	
		public VariantResponse(Variant variant, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/variant/" + variant.getId();
			this.identifier = variant.getIdentifier();
			this.name = variant.getName();
			this.description = variant.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = variant.getChromosome() == null ? null : new ChromosomeResponse(variant.getChromosome(), null);
			else this.chromosome = variant.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/variant/" + variant.getId() + "/chromosome");
			this.cM = variant.getCM();
			this.bpStart = variant.getBpStart();
			this.bpEnd = variant.getBpEnd();
			this.seq = variant.getSeq();
			this.symbol = variant.getSymbol();
			this.mutationPosition = variant.getMutationPosition();
			this.cdnaPosition = variant.getCdnaPosition();
			this.aaPosition = variant.getAaPosition();
			this.variantLength = variant.getVariantLength();
			this.event = variant.getEvent();
			this.ntChange = variant.getNtChange();
			this.codonchange = variant.getCodonchange();
			this.cdnaNotation = variant.getCdnaNotation();
			this.gdnaNotation = variant.getGdnaNotation();
			this.aaNotation = variant.getAaNotation();
			this.exon = variant.getExon();
			this.consequence = variant.getConsequence();
			this.inheritance = variant.getInheritance();
			this.reportedSNP = variant.getReportedSNP();
			this.effectOnSplicing = variant.getEffectOnSplicing();
			this.pathogenicity = variant.getPathogenicity();
			this.gene = variant.getGene();
			this.idMutation = variant.getIdMutation();
			this.detailsForMutation = variant.getDetailsForMutation();
			if (expandFields != null && expandFields.contains("track")) this.track = new TrackResponse(variant.getTrack(), null);
			else this.track = java.util.Collections.singletonMap("href", "/api/v1/variant/" + variant.getId() + "/track");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
		public Integer getMutationPosition()
		{
			return mutationPosition;
		}
	
		public Integer getCdnaPosition()
		{
			return cdnaPosition;
		}
	
		public Integer getAaPosition()
		{
			return aaPosition;
		}
	
		public Integer getVariantLength()
		{
			return variantLength;
		}
	
		public String getEvent()
		{
			return event;
		}
	
		public String getNtChange()
		{
			return ntChange;
		}
	
		public String getCodonchange()
		{
			return codonchange;
		}
	
		public String getCdnaNotation()
		{
			return cdnaNotation;
		}
	
		public String getGdnaNotation()
		{
			return gdnaNotation;
		}
	
		public String getAaNotation()
		{
			return aaNotation;
		}
	
		public String getExon()
		{
			return exon;
		}
	
		public String getConsequence()
		{
			return consequence;
		}
	
		public String getInheritance()
		{
			return inheritance;
		}
	
		public Boolean getReportedSNP()
		{
			return reportedSNP;
		}
	
		public Boolean getEffectOnSplicing()
		{
			return effectOnSplicing;
		}
	
		public String getPathogenicity()
		{
			return pathogenicity;
		}
	
		public String getGene()
		{
			return gene;
		}
	
		public Integer getIdMutation()
		{
			return idMutation;
		}
	
		public String getDetailsForMutation()
		{
			return detailsForMutation;
		}
	
		public Object getTrack()
		{
			return track;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}