package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Clone;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.CloneService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/clone")
public class CloneController
{
	private static Logger logger = Logger.getLogger(CloneController.class);
	 
	@Autowired
	private CloneService cloneService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<CloneResponse> createClone(@Valid @RequestBody CloneRequest cloneRequest)
			throws DatabaseException
	{
		return _createClone(cloneRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CloneResponse> createCloneFromForm(@Valid @ModelAttribute CloneRequest cloneRequest)
			throws DatabaseException
	{
		return _createClone(cloneRequest);
	}

	private ResponseEntity<CloneResponse> _createClone(CloneRequest cloneRequest) throws DatabaseException
	{
		Clone clone = cloneService.create(cloneRequest.toClone());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/clone/" + clone.getId());
		return new ResponseEntity<CloneResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public CloneResponse retrieveClone(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveClone(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public CloneResponse retrieveCloneJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveClone(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private CloneResponse _retrieveClone(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Clone clone = cloneService.read(id);
		if (clone == null) throw new EntityNotFoundException("Clone " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new CloneResponse(clone, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveCloneXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCloneXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveCloneXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCloneXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveCloneXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Clone clone = cloneService.read(id);
		if (clone == null) throw new EntityNotFoundException("Clone " + id.toString() + " not found");
		Integer chromosomeId = clone.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateClone(@PathVariable Integer id, @Valid @RequestBody CloneRequest cloneRequest)
			throws DatabaseException
	{
		_updateClone(id, cloneRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CloneResponse> updateCloneFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute CloneRequest cloneRequest) throws DatabaseException
	{
		return _createClone(cloneRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateClonePost(@PathVariable Integer id, @Valid @RequestBody CloneRequest cloneRequest)
			throws DatabaseException
	{
		_updateClone(id, cloneRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCloneFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute CloneRequest cloneRequest)
			throws DatabaseException
	{
		_updateClone(id, cloneRequest);
	}

	private void _updateClone(Integer id, CloneRequest cloneRequest) throws DatabaseException
	{
		Clone clone = cloneRequest.toClone();
		clone.setId(id);
		cloneService.update(clone);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteClone(@PathVariable Integer id) throws DatabaseException
	{
		_deleteClone(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteClonePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteClone(id);
	}

	private void _deleteClone(Integer id) throws DatabaseException
	{
		boolean isDeleted = cloneService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Clone " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<CloneResponse> retrieveCloneCollection(@Valid EntityCollectionRequest cloneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCloneCollection(cloneCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CloneResponse> retrieveCloneCollectionJson(@Valid EntityCollectionRequest cloneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCloneCollection(cloneCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<CloneResponse> retrieveCloneCollectionPost(@Valid @RequestBody EntityCollectionRequest cloneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCloneCollection(cloneCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CloneResponse> retrieveCloneCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest cloneCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCloneCollection(cloneCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<CloneResponse> _retrieveCloneCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Clone> clonePager = cloneService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<CloneResponse>(clonePager, Lists.newArrayList(Iterables.transform(
				clonePager.getIterable(), new Function<Clone, CloneResponse>()
				{
					@Override
					@Nullable
					public CloneResponse apply(@Nullable Clone clone)
					{
						try
						{
							return clone != null ? new CloneResponse(clone, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/clone");
	}

	private static class CloneRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
	
		public Clone toClone()
		{
			Clone clone = new Clone();
			clone.setIdentifier(identifier);
			clone.setName(name);
			clone.setDescription(description);
			clone.setChromosome_Id(chromosome);
			clone.setCM(cM);
			clone.setBpStart(bpStart);
			clone.setBpEnd(bpEnd);
			clone.setSeq(seq);
			clone.setSymbol(symbol);
			return clone;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
	}

	static class CloneResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
	
		public CloneResponse(Clone clone, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/clone/" + clone.getId();
			this.identifier = clone.getIdentifier();
			this.name = clone.getName();
			this.description = clone.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = clone.getChromosome() == null ? null : new ChromosomeResponse(clone.getChromosome(), null);
			else this.chromosome = clone.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/clone/" + clone.getId() + "/chromosome");
			this.cM = clone.getCM();
			this.bpStart = clone.getBpStart();
			this.bpEnd = clone.getBpEnd();
			this.seq = clone.getSeq();
			this.symbol = clone.getSymbol();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}