package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.value.DecimalValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.DecimalValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/decimalvalue")
public class DecimalValueController
{
	private static Logger logger = Logger.getLogger(DecimalValueController.class);
	 
	@Autowired
	private DecimalValueService decimalValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<DecimalValueResponse> createDecimalValue(@Valid @RequestBody DecimalValueRequest decimalValueRequest)
			throws DatabaseException
	{
		return _createDecimalValue(decimalValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DecimalValueResponse> createDecimalValueFromForm(@Valid @ModelAttribute DecimalValueRequest decimalValueRequest)
			throws DatabaseException
	{
		return _createDecimalValue(decimalValueRequest);
	}

	private ResponseEntity<DecimalValueResponse> _createDecimalValue(DecimalValueRequest decimalValueRequest) throws DatabaseException
	{
		DecimalValue decimalValue = decimalValueService.create(decimalValueRequest.toDecimalValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/decimalvalue/" + decimalValue.getId());
		return new ResponseEntity<DecimalValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public DecimalValueResponse retrieveDecimalValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDecimalValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public DecimalValueResponse retrieveDecimalValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDecimalValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private DecimalValueResponse _retrieveDecimalValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		DecimalValue decimalValue = decimalValueService.read(id);
		if (decimalValue == null) throw new EntityNotFoundException("DecimalValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new DecimalValueResponse(decimalValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateDecimalValue(@PathVariable Integer id, @Valid @RequestBody DecimalValueRequest decimalValueRequest)
			throws DatabaseException
	{
		_updateDecimalValue(id, decimalValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<DecimalValueResponse> updateDecimalValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute DecimalValueRequest decimalValueRequest) throws DatabaseException
	{
		return _createDecimalValue(decimalValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDecimalValuePost(@PathVariable Integer id, @Valid @RequestBody DecimalValueRequest decimalValueRequest)
			throws DatabaseException
	{
		_updateDecimalValue(id, decimalValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateDecimalValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute DecimalValueRequest decimalValueRequest)
			throws DatabaseException
	{
		_updateDecimalValue(id, decimalValueRequest);
	}

	private void _updateDecimalValue(Integer id, DecimalValueRequest decimalValueRequest) throws DatabaseException
	{
		DecimalValue decimalValue = decimalValueRequest.toDecimalValue();
		decimalValue.setId(id);
		decimalValueService.update(decimalValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDecimalValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDecimalValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteDecimalValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteDecimalValue(id);
	}

	private void _deleteDecimalValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = decimalValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("DecimalValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<DecimalValueResponse> retrieveDecimalValueCollection(@Valid EntityCollectionRequest decimalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDecimalValueCollection(decimalValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DecimalValueResponse> retrieveDecimalValueCollectionJson(@Valid EntityCollectionRequest decimalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDecimalValueCollection(decimalValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<DecimalValueResponse> retrieveDecimalValueCollectionPost(@Valid @RequestBody EntityCollectionRequest decimalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDecimalValueCollection(decimalValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<DecimalValueResponse> retrieveDecimalValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest decimalValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveDecimalValueCollection(decimalValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<DecimalValueResponse> _retrieveDecimalValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<DecimalValue> decimalValuePager = decimalValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<DecimalValueResponse>(decimalValuePager, Lists.newArrayList(Iterables.transform(
				decimalValuePager.getIterable(), new Function<DecimalValue, DecimalValueResponse>()
				{
					@Override
					@Nullable
					public DecimalValueResponse apply(@Nullable DecimalValue decimalValue)
					{
						try
						{
							return decimalValue != null ? new DecimalValueResponse(decimalValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/decimalvalue");
	}

	private static class DecimalValueRequest
	{
		private Double value;
	
		public DecimalValue toDecimalValue()
		{
			DecimalValue decimalValue = new DecimalValue();
			decimalValue.setValue(value);
			return decimalValue;
		}
		
		public void setValue(Double value)
		{
			this.value = value;
		}
		
	}

	static class DecimalValueResponse
	{
		private final String href;
		private final Double value;
	
		public DecimalValueResponse(DecimalValue decimalValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/decimalvalue/" + decimalValue.getId();
			this.value = decimalValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public Double getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}