package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.value.HtmlValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.HtmlValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/htmlvalue")
public class HtmlValueController
{
	private static Logger logger = Logger.getLogger(HtmlValueController.class);
	 
	@Autowired
	private HtmlValueService htmlValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<HtmlValueResponse> createHtmlValue(@Valid @RequestBody HtmlValueRequest htmlValueRequest)
			throws DatabaseException
	{
		return _createHtmlValue(htmlValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<HtmlValueResponse> createHtmlValueFromForm(@Valid @ModelAttribute HtmlValueRequest htmlValueRequest)
			throws DatabaseException
	{
		return _createHtmlValue(htmlValueRequest);
	}

	private ResponseEntity<HtmlValueResponse> _createHtmlValue(HtmlValueRequest htmlValueRequest) throws DatabaseException
	{
		HtmlValue htmlValue = htmlValueService.create(htmlValueRequest.toHtmlValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/htmlvalue/" + htmlValue.getId());
		return new ResponseEntity<HtmlValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public HtmlValueResponse retrieveHtmlValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHtmlValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public HtmlValueResponse retrieveHtmlValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHtmlValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private HtmlValueResponse _retrieveHtmlValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		HtmlValue htmlValue = htmlValueService.read(id);
		if (htmlValue == null) throw new EntityNotFoundException("HtmlValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new HtmlValueResponse(htmlValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateHtmlValue(@PathVariable Integer id, @Valid @RequestBody HtmlValueRequest htmlValueRequest)
			throws DatabaseException
	{
		_updateHtmlValue(id, htmlValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<HtmlValueResponse> updateHtmlValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute HtmlValueRequest htmlValueRequest) throws DatabaseException
	{
		return _createHtmlValue(htmlValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateHtmlValuePost(@PathVariable Integer id, @Valid @RequestBody HtmlValueRequest htmlValueRequest)
			throws DatabaseException
	{
		_updateHtmlValue(id, htmlValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateHtmlValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute HtmlValueRequest htmlValueRequest)
			throws DatabaseException
	{
		_updateHtmlValue(id, htmlValueRequest);
	}

	private void _updateHtmlValue(Integer id, HtmlValueRequest htmlValueRequest) throws DatabaseException
	{
		HtmlValue htmlValue = htmlValueRequest.toHtmlValue();
		htmlValue.setId(id);
		htmlValueService.update(htmlValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteHtmlValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteHtmlValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteHtmlValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteHtmlValue(id);
	}

	private void _deleteHtmlValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = htmlValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("HtmlValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<HtmlValueResponse> retrieveHtmlValueCollection(@Valid EntityCollectionRequest htmlValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHtmlValueCollection(htmlValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<HtmlValueResponse> retrieveHtmlValueCollectionJson(@Valid EntityCollectionRequest htmlValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHtmlValueCollection(htmlValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<HtmlValueResponse> retrieveHtmlValueCollectionPost(@Valid @RequestBody EntityCollectionRequest htmlValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHtmlValueCollection(htmlValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<HtmlValueResponse> retrieveHtmlValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest htmlValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveHtmlValueCollection(htmlValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<HtmlValueResponse> _retrieveHtmlValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<HtmlValue> htmlValuePager = htmlValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<HtmlValueResponse>(htmlValuePager, Lists.newArrayList(Iterables.transform(
				htmlValuePager.getIterable(), new Function<HtmlValue, HtmlValueResponse>()
				{
					@Override
					@Nullable
					public HtmlValueResponse apply(@Nullable HtmlValue htmlValue)
					{
						try
						{
							return htmlValue != null ? new HtmlValueResponse(htmlValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/htmlvalue");
	}

	private static class HtmlValueRequest
	{
		private String value;
	
		public HtmlValue toHtmlValue()
		{
			HtmlValue htmlValue = new HtmlValue();
			htmlValue.setValue(value);
			return htmlValue;
		}
		
		public void setValue(String value)
		{
			this.value = value;
		}
		
	}

	static class HtmlValueResponse
	{
		private final String href;
		private final String value;
	
		public HtmlValueResponse(HtmlValue htmlValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/htmlvalue/" + htmlValue.getId();
			this.value = htmlValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public String getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}