package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.target.Individual;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.IndividualService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/individual")
public class IndividualController
{
	private static Logger logger = Logger.getLogger(IndividualController.class);
	 
	@Autowired
	private IndividualService individualService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<IndividualResponse> createIndividual(@Valid @RequestBody IndividualRequest individualRequest)
			throws DatabaseException
	{
		return _createIndividual(individualRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<IndividualResponse> createIndividualFromForm(@Valid @ModelAttribute IndividualRequest individualRequest)
			throws DatabaseException
	{
		return _createIndividual(individualRequest);
	}

	private ResponseEntity<IndividualResponse> _createIndividual(IndividualRequest individualRequest) throws DatabaseException
	{
		Individual individual = individualService.create(individualRequest.toIndividual());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/individual/" + individual.getId());
		return new ResponseEntity<IndividualResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public IndividualResponse retrieveIndividual(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividual(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public IndividualResponse retrieveIndividualJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividual(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private IndividualResponse _retrieveIndividual(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Individual individual = individualService.read(id);
		if (individual == null) throw new EntityNotFoundException("Individual " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new IndividualResponse(individual, expandFields);
	}
			
	@RequestMapping(value = "/{id}/mother", method = RequestMethod.GET)
	public String retrieveIndividualXrefMother(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualXrefMother(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/mother", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveIndividualXrefMotherJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualXrefMother(id, "json", expandFields);
	}
	
	private String _retrieveIndividualXrefMother(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Individual individual = individualService.read(id);
		if (individual == null) throw new EntityNotFoundException("Individual " + id.toString() + " not found");
		Integer individualId = individual.getMother_Id();
		String redirectUri = "redirect:/api/v1/individual/" + individualId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/father", method = RequestMethod.GET)
	public String retrieveIndividualXrefFather(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualXrefFather(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/father", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveIndividualXrefFatherJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualXrefFather(id, "json", expandFields);
	}
	
	private String _retrieveIndividualXrefFather(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Individual individual = individualService.read(id);
		if (individual == null) throw new EntityNotFoundException("Individual " + id.toString() + " not found");
		Integer individualId = individual.getFather_Id();
		String redirectUri = "redirect:/api/v1/individual/" + individualId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateIndividual(@PathVariable Integer id, @Valid @RequestBody IndividualRequest individualRequest)
			throws DatabaseException
	{
		_updateIndividual(id, individualRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<IndividualResponse> updateIndividualFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute IndividualRequest individualRequest) throws DatabaseException
	{
		return _createIndividual(individualRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateIndividualPost(@PathVariable Integer id, @Valid @RequestBody IndividualRequest individualRequest)
			throws DatabaseException
	{
		_updateIndividual(id, individualRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateIndividualFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute IndividualRequest individualRequest)
			throws DatabaseException
	{
		_updateIndividual(id, individualRequest);
	}

	private void _updateIndividual(Integer id, IndividualRequest individualRequest) throws DatabaseException
	{
		Individual individual = individualRequest.toIndividual();
		individual.setId(id);
		individualService.update(individual);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteIndividual(@PathVariable Integer id) throws DatabaseException
	{
		_deleteIndividual(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteIndividualPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteIndividual(id);
	}

	private void _deleteIndividual(Integer id) throws DatabaseException
	{
		boolean isDeleted = individualService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Individual " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<IndividualResponse> retrieveIndividualCollection(@Valid EntityCollectionRequest individualCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualCollection(individualCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<IndividualResponse> retrieveIndividualCollectionJson(@Valid EntityCollectionRequest individualCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualCollection(individualCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<IndividualResponse> retrieveIndividualCollectionPost(@Valid @RequestBody EntityCollectionRequest individualCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualCollection(individualCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<IndividualResponse> retrieveIndividualCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest individualCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIndividualCollection(individualCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<IndividualResponse> _retrieveIndividualCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Individual> individualPager = individualService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<IndividualResponse>(individualPager, Lists.newArrayList(Iterables.transform(
				individualPager.getIterable(), new Function<Individual, IndividualResponse>()
				{
					@Override
					@Nullable
					public IndividualResponse apply(@Nullable Individual individual)
					{
						try
						{
							return individual != null ? new IndividualResponse(individual, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/individual");
	}

	private static class IndividualRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer mother;
		private Integer father;
	
		public Individual toIndividual()
		{
			Individual individual = new Individual();
			individual.setIdentifier(identifier);
			individual.setName(name);
			individual.setDescription(description);
			individual.setMother_Id(mother);
			individual.setFather_Id(father);
			return individual;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setMother(Integer mother)
		{
			this.mother = mother;
		}
		
		public void setFather(Integer father)
		{
			this.father = father;
		}
		
	}

	static class IndividualResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object mother;
		private final Object father;
	
		public IndividualResponse(Individual individual, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/individual/" + individual.getId();
			this.identifier = individual.getIdentifier();
			this.name = individual.getName();
			this.description = individual.getDescription();
			if (expandFields != null && expandFields.contains("mother")) this.mother = individual.getMother() == null ? null : new IndividualResponse(individual.getMother(), null);
			else this.mother = individual.getMother() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/individual/" + individual.getId() + "/mother");
			if (expandFields != null && expandFields.contains("father")) this.father = individual.getFather() == null ? null : new IndividualResponse(individual.getFather(), null);
			else this.father = individual.getFather() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/individual/" + individual.getId() + "/father");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getMother()
		{
			return mother;
		}
	
		public Object getFather()
		{
			return father;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}