package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Marker;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.MarkerService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/marker")
public class MarkerController
{
	private static Logger logger = Logger.getLogger(MarkerController.class);
	 
	@Autowired
	private MarkerService markerService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<MarkerResponse> createMarker(@Valid @RequestBody MarkerRequest markerRequest)
			throws DatabaseException
	{
		return _createMarker(markerRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MarkerResponse> createMarkerFromForm(@Valid @ModelAttribute MarkerRequest markerRequest)
			throws DatabaseException
	{
		return _createMarker(markerRequest);
	}

	private ResponseEntity<MarkerResponse> _createMarker(MarkerRequest markerRequest) throws DatabaseException
	{
		Marker marker = markerService.create(markerRequest.toMarker());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/marker/" + marker.getId());
		return new ResponseEntity<MarkerResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public MarkerResponse retrieveMarker(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarker(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public MarkerResponse retrieveMarkerJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarker(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private MarkerResponse _retrieveMarker(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Marker marker = markerService.read(id);
		if (marker == null) throw new EntityNotFoundException("Marker " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new MarkerResponse(marker, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveMarkerXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarkerXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveMarkerXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarkerXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveMarkerXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Marker marker = markerService.read(id);
		if (marker == null) throw new EntityNotFoundException("Marker " + id.toString() + " not found");
		Integer chromosomeId = marker.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveMarkerMrefReportsFor(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Marker marker = markerService.read(id);
		if (marker == null) throw new EntityNotFoundException("Marker " + id.toString() + " not found");
		return _retrieveMarkerMrefReportsFor(marker, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveMarkerMrefReportsForJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		Marker marker = markerService.read(id);
		if (marker == null) throw new EntityNotFoundException("Marker " + id.toString() + " not found");
		return _retrieveMarkerMrefReportsFor(marker, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<MarkerResponse> _retrieveMarkerMrefReportsFor(Marker marker, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<Marker> markerCollection = marker.getReportsFor();
		
		int total = markerCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		markerCollection = markerCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<Marker> markerPager = new EntityPager<Marker>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, markerCollection);
		
		return new EntityCollectionResponse<MarkerResponse>(markerPager, Lists.newArrayList(Iterables.transform(markerCollection,
				new Function<Marker, MarkerResponse>()
				{
					@Override
					@Nullable
					public MarkerResponse apply(@Nullable Marker marker)
					{
						try
						{
							return marker != null ? new MarkerResponse(marker, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/marker/" + marker.getId() + "/reportsFor");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateMarker(@PathVariable Integer id, @Valid @RequestBody MarkerRequest markerRequest)
			throws DatabaseException
	{
		_updateMarker(id, markerRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MarkerResponse> updateMarkerFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute MarkerRequest markerRequest) throws DatabaseException
	{
		return _createMarker(markerRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMarkerPost(@PathVariable Integer id, @Valid @RequestBody MarkerRequest markerRequest)
			throws DatabaseException
	{
		_updateMarker(id, markerRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMarkerFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute MarkerRequest markerRequest)
			throws DatabaseException
	{
		_updateMarker(id, markerRequest);
	}

	private void _updateMarker(Integer id, MarkerRequest markerRequest) throws DatabaseException
	{
		Marker marker = markerRequest.toMarker();
		marker.setId(id);
		markerService.update(marker);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMarker(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMarker(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMarkerPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMarker(id);
	}

	private void _deleteMarker(Integer id) throws DatabaseException
	{
		boolean isDeleted = markerService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Marker " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveMarkerCollection(@Valid EntityCollectionRequest markerCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarkerCollection(markerCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveMarkerCollectionJson(@Valid EntityCollectionRequest markerCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarkerCollection(markerCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveMarkerCollectionPost(@Valid @RequestBody EntityCollectionRequest markerCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarkerCollection(markerCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveMarkerCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest markerCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMarkerCollection(markerCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<MarkerResponse> _retrieveMarkerCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Marker> markerPager = markerService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<MarkerResponse>(markerPager, Lists.newArrayList(Iterables.transform(
				markerPager.getIterable(), new Function<Marker, MarkerResponse>()
				{
					@Override
					@Nullable
					public MarkerResponse apply(@Nullable Marker marker)
					{
						try
						{
							return marker != null ? new MarkerResponse(marker, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/marker");
	}

	private static class MarkerRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
		private java.util.List<Integer> reportsFor;
	
		public Marker toMarker()
		{
			Marker marker = new Marker();
			marker.setIdentifier(identifier);
			marker.setName(name);
			marker.setDescription(description);
			marker.setChromosome_Id(chromosome);
			marker.setCM(cM);
			marker.setBpStart(bpStart);
			marker.setBpEnd(bpEnd);
			marker.setSeq(seq);
			marker.setSymbol(symbol);
			marker.setReportsFor_Id(reportsFor);
			return marker;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
		public void setReportsFor(java.util.List<Integer> reportsFor)
		{
			this.reportsFor = reportsFor;
		}
		
	}

	static class MarkerResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
		private final Object reportsFor;
	
		public MarkerResponse(Marker marker, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/marker/" + marker.getId();
			this.identifier = marker.getIdentifier();
			this.name = marker.getName();
			this.description = marker.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = marker.getChromosome() == null ? null : new ChromosomeResponse(marker.getChromosome(), null);
			else this.chromosome = marker.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/marker/" + marker.getId() + "/chromosome");
			this.cM = marker.getCM();
			this.bpStart = marker.getBpStart();
			this.bpEnd = marker.getBpEnd();
			this.seq = marker.getSeq();
			this.symbol = marker.getSymbol();
			java.util.List<Marker> ReportsForCollection = marker.getReportsFor();
			if (expandFields != null && expandFields.contains("reportsFor")) this.reportsFor = ReportsForCollection == null ? null : _retrieveMarkerMrefReportsFor(marker, new EntityCollectionRequest());
			else this.reportsFor = ReportsForCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/marker/" + marker.getId() + "/reportsFor");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
		public Object getReportsFor()
		{
			return reportsFor;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}