package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.core.MolgenisEntity;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.MolgenisEntityService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/molgenisentity")
public class MolgenisEntityController
{
	private static Logger logger = Logger.getLogger(MolgenisEntityController.class);
	 
	@Autowired
	private MolgenisEntityService molgenisEntityService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<MolgenisEntityResponse> createMolgenisEntity(@Valid @RequestBody MolgenisEntityRequest molgenisEntityRequest)
			throws DatabaseException
	{
		return _createMolgenisEntity(molgenisEntityRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MolgenisEntityResponse> createMolgenisEntityFromForm(@Valid @ModelAttribute MolgenisEntityRequest molgenisEntityRequest)
			throws DatabaseException
	{
		return _createMolgenisEntity(molgenisEntityRequest);
	}

	private ResponseEntity<MolgenisEntityResponse> _createMolgenisEntity(MolgenisEntityRequest molgenisEntityRequest) throws DatabaseException
	{
		MolgenisEntity molgenisEntity = molgenisEntityService.create(molgenisEntityRequest.toMolgenisEntity());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/molgenisentity/" + molgenisEntity.getId());
		return new ResponseEntity<MolgenisEntityResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public MolgenisEntityResponse retrieveMolgenisEntity(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisEntity(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public MolgenisEntityResponse retrieveMolgenisEntityJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisEntity(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private MolgenisEntityResponse _retrieveMolgenisEntity(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		MolgenisEntity molgenisEntity = molgenisEntityService.read(id);
		if (molgenisEntity == null) throw new EntityNotFoundException("MolgenisEntity " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new MolgenisEntityResponse(molgenisEntity, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateMolgenisEntity(@PathVariable Integer id, @Valid @RequestBody MolgenisEntityRequest molgenisEntityRequest)
			throws DatabaseException
	{
		_updateMolgenisEntity(id, molgenisEntityRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MolgenisEntityResponse> updateMolgenisEntityFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute MolgenisEntityRequest molgenisEntityRequest) throws DatabaseException
	{
		return _createMolgenisEntity(molgenisEntityRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMolgenisEntityPost(@PathVariable Integer id, @Valid @RequestBody MolgenisEntityRequest molgenisEntityRequest)
			throws DatabaseException
	{
		_updateMolgenisEntity(id, molgenisEntityRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMolgenisEntityFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute MolgenisEntityRequest molgenisEntityRequest)
			throws DatabaseException
	{
		_updateMolgenisEntity(id, molgenisEntityRequest);
	}

	private void _updateMolgenisEntity(Integer id, MolgenisEntityRequest molgenisEntityRequest) throws DatabaseException
	{
		MolgenisEntity molgenisEntity = molgenisEntityRequest.toMolgenisEntity();
		molgenisEntity.setId(id);
		molgenisEntityService.update(molgenisEntity);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMolgenisEntity(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMolgenisEntity(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMolgenisEntityPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMolgenisEntity(id);
	}

	private void _deleteMolgenisEntity(Integer id) throws DatabaseException
	{
		boolean isDeleted = molgenisEntityService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("MolgenisEntity " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MolgenisEntityResponse> retrieveMolgenisEntityCollection(@Valid EntityCollectionRequest molgenisEntityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisEntityCollection(molgenisEntityCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MolgenisEntityResponse> retrieveMolgenisEntityCollectionJson(@Valid EntityCollectionRequest molgenisEntityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisEntityCollection(molgenisEntityCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<MolgenisEntityResponse> retrieveMolgenisEntityCollectionPost(@Valid @RequestBody EntityCollectionRequest molgenisEntityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisEntityCollection(molgenisEntityCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MolgenisEntityResponse> retrieveMolgenisEntityCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest molgenisEntityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisEntityCollection(molgenisEntityCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<MolgenisEntityResponse> _retrieveMolgenisEntityCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<MolgenisEntity> molgenisEntityPager = molgenisEntityService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<MolgenisEntityResponse>(molgenisEntityPager, Lists.newArrayList(Iterables.transform(
				molgenisEntityPager.getIterable(), new Function<MolgenisEntity, MolgenisEntityResponse>()
				{
					@Override
					@Nullable
					public MolgenisEntityResponse apply(@Nullable MolgenisEntity molgenisEntity)
					{
						try
						{
							return molgenisEntity != null ? new MolgenisEntityResponse(molgenisEntity, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/molgenisentity");
	}

	private static class MolgenisEntityRequest
	{
		private String name;
		private String type_;
		private String className;
	
		public MolgenisEntity toMolgenisEntity()
		{
			MolgenisEntity molgenisEntity = new MolgenisEntity();
			molgenisEntity.setName(name);
			molgenisEntity.setType(type_);
			molgenisEntity.setClassName(className);
			return molgenisEntity;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setType_(String type_)
		{
			this.type_ = type_;
		}
		
		public void setClassName(String className)
		{
			this.className = className;
		}
		
	}

	static class MolgenisEntityResponse
	{
		private final String href;
		private final String name;
		private final String type_;
		private final String className;
	
		public MolgenisEntityResponse(MolgenisEntity molgenisEntity, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/molgenisentity/" + molgenisEntity.getId();
			this.name = molgenisEntity.getName();
			this.type_ = molgenisEntity.getType();
			this.className = molgenisEntity.getClassName();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getName()
		{
			return name;
		}
	
		public String getType_()
		{
			return type_;
		}
	
		public String getClassName()
		{
			return className;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}