package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.core.MolgenisFile;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.MolgenisFileService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/molgenisfile")
public class MolgenisFileController
{
	private static Logger logger = Logger.getLogger(MolgenisFileController.class);
	 
	@Autowired
	private MolgenisFileService molgenisFileService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<MolgenisFileResponse> createMolgenisFile(@Valid @RequestBody MolgenisFileRequest molgenisFileRequest)
			throws DatabaseException
	{
		return _createMolgenisFile(molgenisFileRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MolgenisFileResponse> createMolgenisFileFromForm(@Valid @ModelAttribute MolgenisFileRequest molgenisFileRequest)
			throws DatabaseException
	{
		return _createMolgenisFile(molgenisFileRequest);
	}

	private ResponseEntity<MolgenisFileResponse> _createMolgenisFile(MolgenisFileRequest molgenisFileRequest) throws DatabaseException
	{
		MolgenisFile molgenisFile = molgenisFileService.create(molgenisFileRequest.toMolgenisFile());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/molgenisfile/" + molgenisFile.getId());
		return new ResponseEntity<MolgenisFileResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public MolgenisFileResponse retrieveMolgenisFile(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisFile(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public MolgenisFileResponse retrieveMolgenisFileJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisFile(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private MolgenisFileResponse _retrieveMolgenisFile(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		MolgenisFile molgenisFile = molgenisFileService.read(id);
		if (molgenisFile == null) throw new EntityNotFoundException("MolgenisFile " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new MolgenisFileResponse(molgenisFile, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateMolgenisFile(@PathVariable Integer id, @Valid @RequestBody MolgenisFileRequest molgenisFileRequest)
			throws DatabaseException
	{
		_updateMolgenisFile(id, molgenisFileRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MolgenisFileResponse> updateMolgenisFileFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute MolgenisFileRequest molgenisFileRequest) throws DatabaseException
	{
		return _createMolgenisFile(molgenisFileRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMolgenisFilePost(@PathVariable Integer id, @Valid @RequestBody MolgenisFileRequest molgenisFileRequest)
			throws DatabaseException
	{
		_updateMolgenisFile(id, molgenisFileRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMolgenisFileFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute MolgenisFileRequest molgenisFileRequest)
			throws DatabaseException
	{
		_updateMolgenisFile(id, molgenisFileRequest);
	}

	private void _updateMolgenisFile(Integer id, MolgenisFileRequest molgenisFileRequest) throws DatabaseException
	{
		MolgenisFile molgenisFile = molgenisFileRequest.toMolgenisFile();
		molgenisFile.setId(id);
		molgenisFileService.update(molgenisFile);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMolgenisFile(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMolgenisFile(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMolgenisFilePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMolgenisFile(id);
	}

	private void _deleteMolgenisFile(Integer id) throws DatabaseException
	{
		boolean isDeleted = molgenisFileService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("MolgenisFile " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MolgenisFileResponse> retrieveMolgenisFileCollection(@Valid EntityCollectionRequest molgenisFileCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisFileCollection(molgenisFileCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MolgenisFileResponse> retrieveMolgenisFileCollectionJson(@Valid EntityCollectionRequest molgenisFileCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisFileCollection(molgenisFileCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<MolgenisFileResponse> retrieveMolgenisFileCollectionPost(@Valid @RequestBody EntityCollectionRequest molgenisFileCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisFileCollection(molgenisFileCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MolgenisFileResponse> retrieveMolgenisFileCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest molgenisFileCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMolgenisFileCollection(molgenisFileCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<MolgenisFileResponse> _retrieveMolgenisFileCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<MolgenisFile> molgenisFilePager = molgenisFileService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<MolgenisFileResponse>(molgenisFilePager, Lists.newArrayList(Iterables.transform(
				molgenisFilePager.getIterable(), new Function<MolgenisFile, MolgenisFileResponse>()
				{
					@Override
					@Nullable
					public MolgenisFileResponse apply(@Nullable MolgenisFile molgenisFile)
					{
						try
						{
							return molgenisFile != null ? new MolgenisFileResponse(molgenisFile, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/molgenisfile");
	}

	private static class MolgenisFileRequest
	{
		private String identifier;
		private String name;
		private String extension;
	
		public MolgenisFile toMolgenisFile()
		{
			MolgenisFile molgenisFile = new MolgenisFile();
			molgenisFile.setIdentifier(identifier);
			molgenisFile.setName(name);
			molgenisFile.setExtension(extension);
			return molgenisFile;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setExtension(String extension)
		{
			this.extension = extension;
		}
		
	}

	static class MolgenisFileResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String extension;
	
		public MolgenisFileResponse(MolgenisFile molgenisFile, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/molgenisfile/" + molgenisFile.getId();
			this.identifier = molgenisFile.getIdentifier();
			this.name = molgenisFile.getName();
			this.extension = molgenisFile.getExtension();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
		public String getExtension()
		{
			return extension;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}