package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.ObservableFeature;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ObservableFeatureService;
import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.controller.OntologyTermController.OntologyTermResponse;
import org.molgenis.service.OntologyTermService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/observablefeature")
public class ObservableFeatureController
{
	private static Logger logger = Logger.getLogger(ObservableFeatureController.class);
	 
	@Autowired
	private ObservableFeatureService observableFeatureService;

	@Autowired
	private OntologyTermService ontologyTermService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ObservableFeatureResponse> createObservableFeature(@Valid @RequestBody ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		return _createObservableFeature(observableFeatureRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservableFeatureResponse> createObservableFeatureFromForm(@Valid @ModelAttribute ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		return _createObservableFeature(observableFeatureRequest);
	}

	private ResponseEntity<ObservableFeatureResponse> _createObservableFeature(ObservableFeatureRequest observableFeatureRequest) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.create(observableFeatureRequest.toObservableFeature());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/observablefeature/" + observableFeature.getId());
		return new ResponseEntity<ObservableFeatureResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ObservableFeatureResponse retrieveObservableFeature(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeature(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ObservableFeatureResponse retrieveObservableFeatureJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeature(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ObservableFeatureResponse _retrieveObservableFeature(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.read(id);
		if (observableFeature == null) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ObservableFeatureResponse(observableFeature, expandFields);
	}
			
	@RequestMapping(value = "/{id}/unit", method = RequestMethod.GET)
	public String retrieveObservableFeatureXrefUnit(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureXrefUnit(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/unit", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveObservableFeatureXrefUnitJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureXrefUnit(id, "json", expandFields);
	}
	
	private String _retrieveObservableFeatureXrefUnit(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.read(id);
		if (observableFeature == null) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
		Integer ontologyTermId = observableFeature.getUnit_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/definitions", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<OntologyTermResponse> retrieveObservableFeatureMrefDefinitions(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.read(id);
		if (observableFeature == null) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
		return _retrieveObservableFeatureMrefDefinitions(observableFeature, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/definitions", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<OntologyTermResponse> retrieveObservableFeatureMrefDefinitionsJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureService.read(id);
		if (observableFeature == null) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
		return _retrieveObservableFeatureMrefDefinitions(observableFeature, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<OntologyTermResponse> _retrieveObservableFeatureMrefDefinitions(ObservableFeature observableFeature, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<OntologyTerm> ontologyTermCollection = observableFeature.getDefinitions();
		
		int total = ontologyTermCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		ontologyTermCollection = ontologyTermCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<OntologyTerm> ontologyTermPager = new EntityPager<OntologyTerm>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, ontologyTermCollection);
		
		return new EntityCollectionResponse<OntologyTermResponse>(ontologyTermPager, Lists.newArrayList(Iterables.transform(ontologyTermCollection,
				new Function<OntologyTerm, OntologyTermResponse>()
				{
					@Override
					@Nullable
					public OntologyTermResponse apply(@Nullable OntologyTerm ontologyTerm)
					{
						try
						{
							return ontologyTerm != null ? new OntologyTermResponse(ontologyTerm, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/observablefeature/" + observableFeature.getId() + "/definitions");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateObservableFeature(@PathVariable Integer id, @Valid @RequestBody ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		_updateObservableFeature(id, observableFeatureRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservableFeatureResponse> updateObservableFeatureFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ObservableFeatureRequest observableFeatureRequest) throws DatabaseException
	{
		return _createObservableFeature(observableFeatureRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservableFeaturePost(@PathVariable Integer id, @Valid @RequestBody ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		_updateObservableFeature(id, observableFeatureRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservableFeatureFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ObservableFeatureRequest observableFeatureRequest)
			throws DatabaseException
	{
		_updateObservableFeature(id, observableFeatureRequest);
	}

	private void _updateObservableFeature(Integer id, ObservableFeatureRequest observableFeatureRequest) throws DatabaseException
	{
		ObservableFeature observableFeature = observableFeatureRequest.toObservableFeature();
		observableFeature.setId(id);
		observableFeatureService.update(observableFeature);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservableFeature(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservableFeature(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservableFeaturePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservableFeature(id);
	}

	private void _deleteObservableFeature(Integer id) throws DatabaseException
	{
		boolean isDeleted = observableFeatureService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("ObservableFeature " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollection(@Valid EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollectionJson(@Valid EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollectionPost(@Valid @RequestBody EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservableFeatureResponse> retrieveObservableFeatureCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest observableFeatureCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservableFeatureCollection(observableFeatureCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ObservableFeatureResponse> _retrieveObservableFeatureCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<ObservableFeature> observableFeaturePager = observableFeatureService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ObservableFeatureResponse>(observableFeaturePager, Lists.newArrayList(Iterables.transform(
				observableFeaturePager.getIterable(), new Function<ObservableFeature, ObservableFeatureResponse>()
				{
					@Override
					@Nullable
					public ObservableFeatureResponse apply(@Nullable ObservableFeature observableFeature)
					{
						try
						{
							return observableFeature != null ? new ObservableFeatureResponse(observableFeature, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/observablefeature");
	}

	private static class ObservableFeatureRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer unit;
		private java.util.List<Integer> definitions;
		private String dataType;
		private Boolean temporal = false;	
	
		public ObservableFeature toObservableFeature()
		{
			ObservableFeature observableFeature = new ObservableFeature();
			observableFeature.setIdentifier(identifier);
			observableFeature.setName(name);
			observableFeature.setDescription(description);
			observableFeature.setUnit_Id(unit);
			observableFeature.setDefinitions_Id(definitions);
			observableFeature.setDataType(dataType);
			observableFeature.setTemporal(temporal);
			return observableFeature;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setUnit(Integer unit)
		{
			this.unit = unit;
		}
		
		public void setDefinitions(java.util.List<Integer> definitions)
		{
			this.definitions = definitions;
		}
		
		public void setDataType(String dataType)
		{
			this.dataType = dataType;
		}
		
		public void setTemporal(Boolean temporal)
		{
			this.temporal = temporal;
		}
		
	}

	static class ObservableFeatureResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object unit;
		private final Object definitions;
		private final String dataType;
		private final Boolean temporal;
	
		public ObservableFeatureResponse(ObservableFeature observableFeature, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/observablefeature/" + observableFeature.getId();
			this.identifier = observableFeature.getIdentifier();
			this.name = observableFeature.getName();
			this.description = observableFeature.getDescription();
			if (expandFields != null && expandFields.contains("unit")) this.unit = observableFeature.getUnit() == null ? null : new OntologyTermResponse(observableFeature.getUnit(), null);
			else this.unit = observableFeature.getUnit() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/observablefeature/" + observableFeature.getId() + "/unit");
			java.util.List<OntologyTerm> definitionsCollection = observableFeature.getDefinitions();
			if (expandFields != null && expandFields.contains("definitions")) this.definitions = definitionsCollection == null ? null : _retrieveObservableFeatureMrefDefinitions(observableFeature, new EntityCollectionRequest());
			else this.definitions = definitionsCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/observablefeature/" + observableFeature.getId() + "/definitions");
			this.dataType = observableFeature.getDataType();
			this.temporal = observableFeature.getTemporal();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getUnit()
		{
			return unit;
		}
	
		public Object getDefinitions()
		{
			return definitions;
		}
	
		public String getDataType()
		{
			return dataType;
		}
	
		public Boolean getTemporal()
		{
			return temporal;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}