package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.ObservationSet;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ObservationSetService;
import org.molgenis.omx.observ.DataSet;
import org.molgenis.controller.DataSetController.DataSetResponse;
import org.molgenis.service.DataSetService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/observationset")
public class ObservationSetController
{
	private static Logger logger = Logger.getLogger(ObservationSetController.class);
	 
	@Autowired
	private ObservationSetService observationSetService;

	@Autowired
	private DataSetService dataSetService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ObservationSetResponse> createObservationSet(@Valid @RequestBody ObservationSetRequest observationSetRequest)
			throws DatabaseException
	{
		return _createObservationSet(observationSetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservationSetResponse> createObservationSetFromForm(@Valid @ModelAttribute ObservationSetRequest observationSetRequest)
			throws DatabaseException
	{
		return _createObservationSet(observationSetRequest);
	}

	private ResponseEntity<ObservationSetResponse> _createObservationSet(ObservationSetRequest observationSetRequest) throws DatabaseException
	{
		ObservationSet observationSet = observationSetService.create(observationSetRequest.toObservationSet());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/observationset/" + observationSet.getId());
		return new ResponseEntity<ObservationSetResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ObservationSetResponse retrieveObservationSet(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSet(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ObservationSetResponse retrieveObservationSetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSet(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ObservationSetResponse _retrieveObservationSet(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		ObservationSet observationSet = observationSetService.read(id);
		if (observationSet == null) throw new EntityNotFoundException("ObservationSet " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ObservationSetResponse(observationSet, expandFields);
	}
			
	@RequestMapping(value = "/{id}/partOfDataSet", method = RequestMethod.GET)
	public String retrieveObservationSetXrefPartOfDataSet(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSetXrefPartOfDataSet(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/partOfDataSet", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveObservationSetXrefPartOfDataSetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSetXrefPartOfDataSet(id, "json", expandFields);
	}
	
	private String _retrieveObservationSetXrefPartOfDataSet(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ObservationSet observationSet = observationSetService.read(id);
		if (observationSet == null) throw new EntityNotFoundException("ObservationSet " + id.toString() + " not found");
		Integer dataSetId = observationSet.getPartOfDataSet_Id();
		String redirectUri = "redirect:/api/v1/dataset/" + dataSetId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateObservationSet(@PathVariable Integer id, @Valid @RequestBody ObservationSetRequest observationSetRequest)
			throws DatabaseException
	{
		_updateObservationSet(id, observationSetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ObservationSetResponse> updateObservationSetFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ObservationSetRequest observationSetRequest) throws DatabaseException
	{
		return _createObservationSet(observationSetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservationSetPost(@PathVariable Integer id, @Valid @RequestBody ObservationSetRequest observationSetRequest)
			throws DatabaseException
	{
		_updateObservationSet(id, observationSetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateObservationSetFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ObservationSetRequest observationSetRequest)
			throws DatabaseException
	{
		_updateObservationSet(id, observationSetRequest);
	}

	private void _updateObservationSet(Integer id, ObservationSetRequest observationSetRequest) throws DatabaseException
	{
		ObservationSet observationSet = observationSetRequest.toObservationSet();
		observationSet.setId(id);
		observationSetService.update(observationSet);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservationSet(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservationSet(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteObservationSetPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteObservationSet(id);
	}

	private void _deleteObservationSet(Integer id) throws DatabaseException
	{
		boolean isDeleted = observationSetService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("ObservationSet " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ObservationSetResponse> retrieveObservationSetCollection(@Valid EntityCollectionRequest observationSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSetCollection(observationSetCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservationSetResponse> retrieveObservationSetCollectionJson(@Valid EntityCollectionRequest observationSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSetCollection(observationSetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ObservationSetResponse> retrieveObservationSetCollectionPost(@Valid @RequestBody EntityCollectionRequest observationSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSetCollection(observationSetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ObservationSetResponse> retrieveObservationSetCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest observationSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveObservationSetCollection(observationSetCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ObservationSetResponse> _retrieveObservationSetCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<ObservationSet> observationSetPager = observationSetService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ObservationSetResponse>(observationSetPager, Lists.newArrayList(Iterables.transform(
				observationSetPager.getIterable(), new Function<ObservationSet, ObservationSetResponse>()
				{
					@Override
					@Nullable
					public ObservationSetResponse apply(@Nullable ObservationSet observationSet)
					{
						try
						{
							return observationSet != null ? new ObservationSetResponse(observationSet, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/observationset");
	}

	private static class ObservationSetRequest
	{
		private String identifier;
		private Integer partOfDataSet;
		private java.util.Date time;
	
		public ObservationSet toObservationSet()
		{
			ObservationSet observationSet = new ObservationSet();
			observationSet.setIdentifier(identifier);
			observationSet.setPartOfDataSet_Id(partOfDataSet);
			observationSet.setTime(time);
			return observationSet;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setPartOfDataSet(Integer partOfDataSet)
		{
			this.partOfDataSet = partOfDataSet;
		}
		
		public void setTime(java.util.Date time)
		{
			this.time = time;
		}
		
	}

	static class ObservationSetResponse
	{
		private final String href;
		private final String identifier;
		private final Object partOfDataSet;
		private final java.util.Date time;
	
		public ObservationSetResponse(ObservationSet observationSet, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/observationset/" + observationSet.getId();
			this.identifier = observationSet.getIdentifier();
			if (expandFields != null && expandFields.contains("partOfDataSet")) this.partOfDataSet = new DataSetResponse(observationSet.getPartOfDataSet(), null);
			else this.partOfDataSet = java.util.Collections.singletonMap("href", "/api/v1/observationset/" + observationSet.getId() + "/partOfDataSet");
			this.time = observationSet.getTime();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public Object getPartOfDataSet()
		{
			return partOfDataSet;
		}
	
		public java.util.Date getTime()
		{
			return time;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}