package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.OntologyTermService;
import org.molgenis.omx.observ.target.Ontology;
import org.molgenis.controller.OntologyController.OntologyResponse;
import org.molgenis.service.OntologyService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/ontologyterm")
public class OntologyTermController
{
	private static Logger logger = Logger.getLogger(OntologyTermController.class);
	 
	@Autowired
	private OntologyTermService ontologyTermService;

	@Autowired
	private OntologyService ontologyService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<OntologyTermResponse> createOntologyTerm(@Valid @RequestBody OntologyTermRequest ontologyTermRequest)
			throws DatabaseException
	{
		return _createOntologyTerm(ontologyTermRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<OntologyTermResponse> createOntologyTermFromForm(@Valid @ModelAttribute OntologyTermRequest ontologyTermRequest)
			throws DatabaseException
	{
		return _createOntologyTerm(ontologyTermRequest);
	}

	private ResponseEntity<OntologyTermResponse> _createOntologyTerm(OntologyTermRequest ontologyTermRequest) throws DatabaseException
	{
		OntologyTerm ontologyTerm = ontologyTermService.create(ontologyTermRequest.toOntologyTerm());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/ontologyterm/" + ontologyTerm.getId());
		return new ResponseEntity<OntologyTermResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public OntologyTermResponse retrieveOntologyTerm(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTerm(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public OntologyTermResponse retrieveOntologyTermJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTerm(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private OntologyTermResponse _retrieveOntologyTerm(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		OntologyTerm ontologyTerm = ontologyTermService.read(id);
		if (ontologyTerm == null) throw new EntityNotFoundException("OntologyTerm " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new OntologyTermResponse(ontologyTerm, expandFields);
	}
			
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET)
	public String retrieveOntologyTermXrefOntology(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTermXrefOntology(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/ontology", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveOntologyTermXrefOntologyJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTermXrefOntology(id, "json", expandFields);
	}
	
	private String _retrieveOntologyTermXrefOntology(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		OntologyTerm ontologyTerm = ontologyTermService.read(id);
		if (ontologyTerm == null) throw new EntityNotFoundException("OntologyTerm " + id.toString() + " not found");
		Integer ontologyId = ontologyTerm.getOntology_Id();
		String redirectUri = "redirect:/api/v1/ontology/" + ontologyId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateOntologyTerm(@PathVariable Integer id, @Valid @RequestBody OntologyTermRequest ontologyTermRequest)
			throws DatabaseException
	{
		_updateOntologyTerm(id, ontologyTermRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<OntologyTermResponse> updateOntologyTermFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute OntologyTermRequest ontologyTermRequest) throws DatabaseException
	{
		return _createOntologyTerm(ontologyTermRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateOntologyTermPost(@PathVariable Integer id, @Valid @RequestBody OntologyTermRequest ontologyTermRequest)
			throws DatabaseException
	{
		_updateOntologyTerm(id, ontologyTermRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateOntologyTermFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute OntologyTermRequest ontologyTermRequest)
			throws DatabaseException
	{
		_updateOntologyTerm(id, ontologyTermRequest);
	}

	private void _updateOntologyTerm(Integer id, OntologyTermRequest ontologyTermRequest) throws DatabaseException
	{
		OntologyTerm ontologyTerm = ontologyTermRequest.toOntologyTerm();
		ontologyTerm.setId(id);
		ontologyTermService.update(ontologyTerm);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteOntologyTerm(@PathVariable Integer id) throws DatabaseException
	{
		_deleteOntologyTerm(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteOntologyTermPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteOntologyTerm(id);
	}

	private void _deleteOntologyTerm(Integer id) throws DatabaseException
	{
		boolean isDeleted = ontologyTermService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("OntologyTerm " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<OntologyTermResponse> retrieveOntologyTermCollection(@Valid EntityCollectionRequest ontologyTermCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTermCollection(ontologyTermCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<OntologyTermResponse> retrieveOntologyTermCollectionJson(@Valid EntityCollectionRequest ontologyTermCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTermCollection(ontologyTermCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<OntologyTermResponse> retrieveOntologyTermCollectionPost(@Valid @RequestBody EntityCollectionRequest ontologyTermCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTermCollection(ontologyTermCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<OntologyTermResponse> retrieveOntologyTermCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest ontologyTermCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveOntologyTermCollection(ontologyTermCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<OntologyTermResponse> _retrieveOntologyTermCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<OntologyTerm> ontologyTermPager = ontologyTermService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<OntologyTermResponse>(ontologyTermPager, Lists.newArrayList(Iterables.transform(
				ontologyTermPager.getIterable(), new Function<OntologyTerm, OntologyTermResponse>()
				{
					@Override
					@Nullable
					public OntologyTermResponse apply(@Nullable OntologyTerm ontologyTerm)
					{
						try
						{
							return ontologyTerm != null ? new OntologyTermResponse(ontologyTerm, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/ontologyterm");
	}

	private static class OntologyTermRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer ontology;
		private String termAccession;
		private String definition;
	
		public OntologyTerm toOntologyTerm()
		{
			OntologyTerm ontologyTerm = new OntologyTerm();
			ontologyTerm.setIdentifier(identifier);
			ontologyTerm.setName(name);
			ontologyTerm.setDescription(description);
			ontologyTerm.setOntology_Id(ontology);
			ontologyTerm.setTermAccession(termAccession);
			ontologyTerm.setDefinition(definition);
			return ontologyTerm;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setOntology(Integer ontology)
		{
			this.ontology = ontology;
		}
		
		public void setTermAccession(String termAccession)
		{
			this.termAccession = termAccession;
		}
		
		public void setDefinition(String definition)
		{
			this.definition = definition;
		}
		
	}

	static class OntologyTermResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object ontology;
		private final String termAccession;
		private final String definition;
	
		public OntologyTermResponse(OntologyTerm ontologyTerm, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/ontologyterm/" + ontologyTerm.getId();
			this.identifier = ontologyTerm.getIdentifier();
			this.name = ontologyTerm.getName();
			this.description = ontologyTerm.getDescription();
			if (expandFields != null && expandFields.contains("ontology")) this.ontology = ontologyTerm.getOntology() == null ? null : new OntologyResponse(ontologyTerm.getOntology(), null);
			else this.ontology = ontologyTerm.getOntology() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/ontologyterm/" + ontologyTerm.getId() + "/ontology");
			this.termAccession = ontologyTerm.getTermAccession();
			this.definition = ontologyTerm.getDefinition();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getOntology()
		{
			return ontology;
		}
	
		public String getTermAccession()
		{
			return termAccession;
		}
	
		public String getDefinition()
		{
			return definition;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}