package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.patient.Patient;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.PatientService;
import org.molgenis.omx.observ.target.Individual;
import org.molgenis.controller.IndividualController.IndividualResponse;
import org.molgenis.service.IndividualService;
import org.molgenis.omx.xgap.Variant;
import org.molgenis.controller.VariantController.VariantResponse;
import org.molgenis.service.VariantService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/patient")
public class PatientController
{
	private static Logger logger = Logger.getLogger(PatientController.class);
	 
	@Autowired
	private PatientService patientService;

	@Autowired
	private IndividualService individualService;
		
	@Autowired
	private VariantService variantService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<PatientResponse> createPatient(@Valid @RequestBody PatientRequest patientRequest)
			throws DatabaseException
	{
		return _createPatient(patientRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PatientResponse> createPatientFromForm(@Valid @ModelAttribute PatientRequest patientRequest)
			throws DatabaseException
	{
		return _createPatient(patientRequest);
	}

	private ResponseEntity<PatientResponse> _createPatient(PatientRequest patientRequest) throws DatabaseException
	{
		Patient patient = patientService.create(patientRequest.toPatient());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/patient/" + patient.getId());
		return new ResponseEntity<PatientResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public PatientResponse retrievePatient(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatient(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public PatientResponse retrievePatientJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatient(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private PatientResponse _retrievePatient(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Patient patient = patientService.read(id);
		if (patient == null) throw new EntityNotFoundException("Patient " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new PatientResponse(patient, expandFields);
	}
			
	@RequestMapping(value = "/{id}/mother", method = RequestMethod.GET)
	public String retrievePatientXrefMother(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefMother(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/mother", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePatientXrefMotherJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefMother(id, "json", expandFields);
	}
	
	private String _retrievePatientXrefMother(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Patient patient = patientService.read(id);
		if (patient == null) throw new EntityNotFoundException("Patient " + id.toString() + " not found");
		Integer individualId = patient.getMother_Id();
		String redirectUri = "redirect:/api/v1/individual/" + individualId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/father", method = RequestMethod.GET)
	public String retrievePatientXrefFather(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefFather(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/father", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePatientXrefFatherJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefFather(id, "json", expandFields);
	}
	
	private String _retrievePatientXrefFather(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Patient patient = patientService.read(id);
		if (patient == null) throw new EntityNotFoundException("Patient " + id.toString() + " not found");
		Integer individualId = patient.getFather_Id();
		String redirectUri = "redirect:/api/v1/individual/" + individualId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/allele1", method = RequestMethod.GET)
	public String retrievePatientXrefAllele1(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefAllele1(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/allele1", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePatientXrefAllele1Json(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefAllele1(id, "json", expandFields);
	}
	
	private String _retrievePatientXrefAllele1(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Patient patient = patientService.read(id);
		if (patient == null) throw new EntityNotFoundException("Patient " + id.toString() + " not found");
		Integer variantId = patient.getAllele1_Id();
		String redirectUri = "redirect:/api/v1/variant/" + variantId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/allele2", method = RequestMethod.GET)
	public String retrievePatientXrefAllele2(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefAllele2(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/allele2", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePatientXrefAllele2Json(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientXrefAllele2(id, "json", expandFields);
	}
	
	private String _retrievePatientXrefAllele2(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Patient patient = patientService.read(id);
		if (patient == null) throw new EntityNotFoundException("Patient " + id.toString() + " not found");
		Integer variantId = patient.getAllele2_Id();
		String redirectUri = "redirect:/api/v1/variant/" + variantId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updatePatient(@PathVariable Integer id, @Valid @RequestBody PatientRequest patientRequest)
			throws DatabaseException
	{
		_updatePatient(id, patientRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PatientResponse> updatePatientFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute PatientRequest patientRequest) throws DatabaseException
	{
		return _createPatient(patientRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePatientPost(@PathVariable Integer id, @Valid @RequestBody PatientRequest patientRequest)
			throws DatabaseException
	{
		_updatePatient(id, patientRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePatientFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute PatientRequest patientRequest)
			throws DatabaseException
	{
		_updatePatient(id, patientRequest);
	}

	private void _updatePatient(Integer id, PatientRequest patientRequest) throws DatabaseException
	{
		Patient patient = patientRequest.toPatient();
		patient.setId(id);
		patientService.update(patient);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePatient(@PathVariable Integer id) throws DatabaseException
	{
		_deletePatient(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePatientPost(@PathVariable Integer id) throws DatabaseException
	{
		_deletePatient(id);
	}

	private void _deletePatient(Integer id) throws DatabaseException
	{
		boolean isDeleted = patientService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Patient " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PatientResponse> retrievePatientCollection(@Valid EntityCollectionRequest patientCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientCollection(patientCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PatientResponse> retrievePatientCollectionJson(@Valid EntityCollectionRequest patientCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientCollection(patientCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<PatientResponse> retrievePatientCollectionPost(@Valid @RequestBody EntityCollectionRequest patientCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientCollection(patientCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PatientResponse> retrievePatientCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest patientCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePatientCollection(patientCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<PatientResponse> _retrievePatientCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Patient> patientPager = patientService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<PatientResponse>(patientPager, Lists.newArrayList(Iterables.transform(
				patientPager.getIterable(), new Function<Patient, PatientResponse>()
				{
					@Override
					@Nullable
					public PatientResponse apply(@Nullable Patient patient)
					{
						try
						{
							return patient != null ? new PatientResponse(patient, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/patient");
	}

	private static class PatientRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer mother;
		private Integer father;
		private Integer allele1;
		private Integer allele2;
		private String pheno;
		private String pubMedID;
		private String reference;
	
		public Patient toPatient()
		{
			Patient patient = new Patient();
			patient.setIdentifier(identifier);
			patient.setName(name);
			patient.setDescription(description);
			patient.setMother_Id(mother);
			patient.setFather_Id(father);
			patient.setAllele1_Id(allele1);
			patient.setAllele2_Id(allele2);
			patient.setPheno(pheno);
			patient.setPubMedID(pubMedID);
			patient.setReference(reference);
			return patient;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setMother(Integer mother)
		{
			this.mother = mother;
		}
		
		public void setFather(Integer father)
		{
			this.father = father;
		}
		
		public void setAllele1(Integer allele1)
		{
			this.allele1 = allele1;
		}
		
		public void setAllele2(Integer allele2)
		{
			this.allele2 = allele2;
		}
		
		public void setPheno(String pheno)
		{
			this.pheno = pheno;
		}
		
		public void setPubMedID(String pubMedID)
		{
			this.pubMedID = pubMedID;
		}
		
		public void setReference(String reference)
		{
			this.reference = reference;
		}
		
	}

	static class PatientResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object mother;
		private final Object father;
		private final Object allele1;
		private final Object allele2;
		private final String pheno;
		private final String pubMedID;
		private final String reference;
	
		public PatientResponse(Patient patient, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/patient/" + patient.getId();
			this.identifier = patient.getIdentifier();
			this.name = patient.getName();
			this.description = patient.getDescription();
			if (expandFields != null && expandFields.contains("mother")) this.mother = patient.getMother() == null ? null : new IndividualResponse(patient.getMother(), null);
			else this.mother = patient.getMother() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/patient/" + patient.getId() + "/mother");
			if (expandFields != null && expandFields.contains("father")) this.father = patient.getFather() == null ? null : new IndividualResponse(patient.getFather(), null);
			else this.father = patient.getFather() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/patient/" + patient.getId() + "/father");
			if (expandFields != null && expandFields.contains("allele1")) this.allele1 = patient.getAllele1() == null ? null : new VariantResponse(patient.getAllele1(), null);
			else this.allele1 = patient.getAllele1() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/patient/" + patient.getId() + "/allele1");
			if (expandFields != null && expandFields.contains("allele2")) this.allele2 = patient.getAllele2() == null ? null : new VariantResponse(patient.getAllele2(), null);
			else this.allele2 = patient.getAllele2() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/patient/" + patient.getId() + "/allele2");
			this.pheno = patient.getPheno();
			this.pubMedID = patient.getPubMedID();
			this.reference = patient.getReference();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getMother()
		{
			return mother;
		}
	
		public Object getFather()
		{
			return father;
		}
	
		public Object getAllele1()
		{
			return allele1;
		}
	
		public Object getAllele2()
		{
			return allele2;
		}
	
		public String getPheno()
		{
			return pheno;
		}
	
		public String getPubMedID()
		{
			return pubMedID;
		}
	
		public String getReference()
		{
			return reference;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}