package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.auth.Person;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.PersonService;
import org.molgenis.omx.auth.Institute;
import org.molgenis.controller.InstituteController.InstituteResponse;
import org.molgenis.service.InstituteService;
import org.molgenis.omx.auth.PersonRole;
import org.molgenis.controller.PersonRoleController.PersonRoleResponse;
import org.molgenis.service.PersonRoleService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/person")
public class PersonController
{
	private static Logger logger = Logger.getLogger(PersonController.class);
	 
	@Autowired
	private PersonService personService;

	@Autowired
	private InstituteService instituteService;
		
	@Autowired
	private PersonRoleService personRoleService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<PersonResponse> createPerson(@Valid @RequestBody PersonRequest personRequest)
			throws DatabaseException
	{
		return _createPerson(personRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PersonResponse> createPersonFromForm(@Valid @ModelAttribute PersonRequest personRequest)
			throws DatabaseException
	{
		return _createPerson(personRequest);
	}

	private ResponseEntity<PersonResponse> _createPerson(PersonRequest personRequest) throws DatabaseException
	{
		Person person = personService.create(personRequest.toPerson());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/person/" + person.getId());
		return new ResponseEntity<PersonResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public PersonResponse retrievePerson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePerson(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public PersonResponse retrievePersonJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePerson(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private PersonResponse _retrievePerson(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Person person = personService.read(id);
		if (person == null) throw new EntityNotFoundException("Person " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new PersonResponse(person, expandFields);
	}
			
	@RequestMapping(value = "/{id}/affiliation", method = RequestMethod.GET)
	public String retrievePersonXrefAffiliation(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonXrefAffiliation(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/affiliation", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePersonXrefAffiliationJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonXrefAffiliation(id, "json", expandFields);
	}
	
	private String _retrievePersonXrefAffiliation(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Person person = personService.read(id);
		if (person == null) throw new EntityNotFoundException("Person " + id.toString() + " not found");
		Integer instituteId = person.getAffiliation_Id();
		String redirectUri = "redirect:/api/v1/institute/" + instituteId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/roles", method = RequestMethod.GET)
	public String retrievePersonXrefRoles(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonXrefRoles(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/roles", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrievePersonXrefRolesJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonXrefRoles(id, "json", expandFields);
	}
	
	private String _retrievePersonXrefRoles(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Person person = personService.read(id);
		if (person == null) throw new EntityNotFoundException("Person " + id.toString() + " not found");
		Integer personRoleId = person.getRoles_Id();
		String redirectUri = "redirect:/api/v1/personrole/" + personRoleId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updatePerson(@PathVariable Integer id, @Valid @RequestBody PersonRequest personRequest)
			throws DatabaseException
	{
		_updatePerson(id, personRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PersonResponse> updatePersonFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute PersonRequest personRequest) throws DatabaseException
	{
		return _createPerson(personRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePersonPost(@PathVariable Integer id, @Valid @RequestBody PersonRequest personRequest)
			throws DatabaseException
	{
		_updatePerson(id, personRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePersonFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute PersonRequest personRequest)
			throws DatabaseException
	{
		_updatePerson(id, personRequest);
	}

	private void _updatePerson(Integer id, PersonRequest personRequest) throws DatabaseException
	{
		Person person = personRequest.toPerson();
		person.setId(id);
		personService.update(person);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePerson(@PathVariable Integer id) throws DatabaseException
	{
		_deletePerson(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePersonPost(@PathVariable Integer id) throws DatabaseException
	{
		_deletePerson(id);
	}

	private void _deletePerson(Integer id) throws DatabaseException
	{
		boolean isDeleted = personService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Person " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PersonResponse> retrievePersonCollection(@Valid EntityCollectionRequest personCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonCollection(personCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PersonResponse> retrievePersonCollectionJson(@Valid EntityCollectionRequest personCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonCollection(personCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<PersonResponse> retrievePersonCollectionPost(@Valid @RequestBody EntityCollectionRequest personCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonCollection(personCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PersonResponse> retrievePersonCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest personCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePersonCollection(personCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<PersonResponse> _retrievePersonCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Person> personPager = personService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<PersonResponse>(personPager, Lists.newArrayList(Iterables.transform(
				personPager.getIterable(), new Function<Person, PersonResponse>()
				{
					@Override
					@Nullable
					public PersonResponse apply(@Nullable Person person)
					{
						try
						{
							return person != null ? new PersonResponse(person, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/person");
	}

	private static class PersonRequest
	{
		private String identifier;
		private String name;
		private String description;
		private String address;
		private String phone;
		private String email;
		private String fax;
		private String tollFreePhone;
		private String city;
		private String country;
		private String firstName;
		private String midInitials;
		private String lastName;
		private String title;
		private Integer affiliation;
		private String department;
		private Integer roles;
	
		public Person toPerson()
		{
			Person person = new Person();
			person.setIdentifier(identifier);
			person.setName(name);
			person.setDescription(description);
			person.setAddress(address);
			person.setPhone(phone);
			person.setEmail(email);
			person.setFax(fax);
			person.setTollFreePhone(tollFreePhone);
			person.setCity(city);
			person.setCountry(country);
			person.setFirstName(firstName);
			person.setMidInitials(midInitials);
			person.setLastName(lastName);
			person.setTitle(title);
			person.setAffiliation_Id(affiliation);
			person.setDepartment(department);
			person.setRoles_Id(roles);
			return person;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setAddress(String address)
		{
			this.address = address;
		}
		
		public void setPhone(String phone)
		{
			this.phone = phone;
		}
		
		public void setEmail(String email)
		{
			this.email = email;
		}
		
		public void setFax(String fax)
		{
			this.fax = fax;
		}
		
		public void setTollFreePhone(String tollFreePhone)
		{
			this.tollFreePhone = tollFreePhone;
		}
		
		public void setCity(String city)
		{
			this.city = city;
		}
		
		public void setCountry(String country)
		{
			this.country = country;
		}
		
		public void setFirstName(String firstName)
		{
			this.firstName = firstName;
		}
		
		public void setMidInitials(String midInitials)
		{
			this.midInitials = midInitials;
		}
		
		public void setLastName(String lastName)
		{
			this.lastName = lastName;
		}
		
		public void setTitle(String title)
		{
			this.title = title;
		}
		
		public void setAffiliation(Integer affiliation)
		{
			this.affiliation = affiliation;
		}
		
		public void setDepartment(String department)
		{
			this.department = department;
		}
		
		public void setRoles(Integer roles)
		{
			this.roles = roles;
		}
		
	}

	static class PersonResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final String address;
		private final String phone;
		private final String email;
		private final String fax;
		private final String tollFreePhone;
		private final String city;
		private final String country;
		private final String firstName;
		private final String midInitials;
		private final String lastName;
		private final String title;
		private final Object affiliation;
		private final String department;
		private final Object roles;
	
		public PersonResponse(Person person, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/person/" + person.getId();
			this.identifier = person.getIdentifier();
			this.name = person.getName();
			this.description = person.getDescription();
			this.address = person.getAddress();
			this.phone = person.getPhone();
			this.email = person.getEmail();
			this.fax = person.getFax();
			this.tollFreePhone = person.getTollFreePhone();
			this.city = person.getCity();
			this.country = person.getCountry();
			this.firstName = person.getFirstName();
			this.midInitials = person.getMidInitials();
			this.lastName = person.getLastName();
			this.title = person.getTitle();
			if (expandFields != null && expandFields.contains("affiliation")) this.affiliation = person.getAffiliation() == null ? null : new InstituteResponse(person.getAffiliation(), null);
			else this.affiliation = person.getAffiliation() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/person/" + person.getId() + "/affiliation");
			this.department = person.getDepartment();
			if (expandFields != null && expandFields.contains("roles")) this.roles = person.getRoles() == null ? null : new PersonRoleResponse(person.getRoles(), null);
			else this.roles = person.getRoles() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/person/" + person.getId() + "/roles");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public String getAddress()
		{
			return address;
		}
	
		public String getPhone()
		{
			return phone;
		}
	
		public String getEmail()
		{
			return email;
		}
	
		public String getFax()
		{
			return fax;
		}
	
		public String getTollFreePhone()
		{
			return tollFreePhone;
		}
	
		public String getCity()
		{
			return city;
		}
	
		public String getCountry()
		{
			return country;
		}
	
		public String getFirstName()
		{
			return firstName;
		}
	
		public String getMidInitials()
		{
			return midInitials;
		}
	
		public String getLastName()
		{
			return lastName;
		}
	
		public String getTitle()
		{
			return title;
		}
	
		public Object getAffiliation()
		{
			return affiliation;
		}
	
		public String getDepartment()
		{
			return department;
		}
	
		public Object getRoles()
		{
			return roles;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}