package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Polymorphism;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.PolymorphismService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/polymorphism")
public class PolymorphismController
{
	private static Logger logger = Logger.getLogger(PolymorphismController.class);
	 
	@Autowired
	private PolymorphismService polymorphismService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<PolymorphismResponse> createPolymorphism(@Valid @RequestBody PolymorphismRequest polymorphismRequest)
			throws DatabaseException
	{
		return _createPolymorphism(polymorphismRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PolymorphismResponse> createPolymorphismFromForm(@Valid @ModelAttribute PolymorphismRequest polymorphismRequest)
			throws DatabaseException
	{
		return _createPolymorphism(polymorphismRequest);
	}

	private ResponseEntity<PolymorphismResponse> _createPolymorphism(PolymorphismRequest polymorphismRequest) throws DatabaseException
	{
		Polymorphism polymorphism = polymorphismService.create(polymorphismRequest.toPolymorphism());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/polymorphism/" + polymorphism.getId());
		return new ResponseEntity<PolymorphismResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public PolymorphismResponse retrievePolymorphism(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePolymorphism(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public PolymorphismResponse retrievePolymorphismJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePolymorphism(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private PolymorphismResponse _retrievePolymorphism(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Polymorphism polymorphism = polymorphismService.read(id);
		if (polymorphism == null) throw new EntityNotFoundException("Polymorphism " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new PolymorphismResponse(polymorphism, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updatePolymorphism(@PathVariable Integer id, @Valid @RequestBody PolymorphismRequest polymorphismRequest)
			throws DatabaseException
	{
		_updatePolymorphism(id, polymorphismRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<PolymorphismResponse> updatePolymorphismFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute PolymorphismRequest polymorphismRequest) throws DatabaseException
	{
		return _createPolymorphism(polymorphismRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePolymorphismPost(@PathVariable Integer id, @Valid @RequestBody PolymorphismRequest polymorphismRequest)
			throws DatabaseException
	{
		_updatePolymorphism(id, polymorphismRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updatePolymorphismFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute PolymorphismRequest polymorphismRequest)
			throws DatabaseException
	{
		_updatePolymorphism(id, polymorphismRequest);
	}

	private void _updatePolymorphism(Integer id, PolymorphismRequest polymorphismRequest) throws DatabaseException
	{
		Polymorphism polymorphism = polymorphismRequest.toPolymorphism();
		polymorphism.setId(id);
		polymorphismService.update(polymorphism);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePolymorphism(@PathVariable Integer id) throws DatabaseException
	{
		_deletePolymorphism(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deletePolymorphismPost(@PathVariable Integer id) throws DatabaseException
	{
		_deletePolymorphism(id);
	}

	private void _deletePolymorphism(Integer id) throws DatabaseException
	{
		boolean isDeleted = polymorphismService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Polymorphism " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PolymorphismResponse> retrievePolymorphismCollection(@Valid EntityCollectionRequest polymorphismCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePolymorphismCollection(polymorphismCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PolymorphismResponse> retrievePolymorphismCollectionJson(@Valid EntityCollectionRequest polymorphismCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePolymorphismCollection(polymorphismCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<PolymorphismResponse> retrievePolymorphismCollectionPost(@Valid @RequestBody EntityCollectionRequest polymorphismCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePolymorphismCollection(polymorphismCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PolymorphismResponse> retrievePolymorphismCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest polymorphismCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrievePolymorphismCollection(polymorphismCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<PolymorphismResponse> _retrievePolymorphismCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Polymorphism> polymorphismPager = polymorphismService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<PolymorphismResponse>(polymorphismPager, Lists.newArrayList(Iterables.transform(
				polymorphismPager.getIterable(), new Function<Polymorphism, PolymorphismResponse>()
				{
					@Override
					@Nullable
					public PolymorphismResponse apply(@Nullable Polymorphism polymorphism)
					{
						try
						{
							return polymorphism != null ? new PolymorphismResponse(polymorphism, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/polymorphism");
	}

	private static class PolymorphismRequest
	{
		private String identifier;
		private String name;
		private String description;
		private String base;
		private String value;
	
		public Polymorphism toPolymorphism()
		{
			Polymorphism polymorphism = new Polymorphism();
			polymorphism.setIdentifier(identifier);
			polymorphism.setName(name);
			polymorphism.setDescription(description);
			polymorphism.setBase(base);
			polymorphism.setValue(value);
			return polymorphism;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setBase(String base)
		{
			this.base = base;
		}
		
		public void setValue(String value)
		{
			this.value = value;
		}
		
	}

	static class PolymorphismResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final String base;
		private final String value;
	
		public PolymorphismResponse(Polymorphism polymorphism, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/polymorphism/" + polymorphism.getId();
			this.identifier = polymorphism.getIdentifier();
			this.name = polymorphism.getName();
			this.description = polymorphism.getDescription();
			this.base = polymorphism.getBase();
			this.value = polymorphism.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public String getBase()
		{
			return base;
		}
	
		public String getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}