package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Probe;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ProbeService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.omx.xgap.ProbeSet;
import org.molgenis.controller.ProbeSetController.ProbeSetResponse;
import org.molgenis.service.ProbeSetService;
import org.molgenis.omx.xgap.Gene;
import org.molgenis.controller.GeneController.GeneResponse;
import org.molgenis.service.GeneService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/probe")
public class ProbeController
{
	private static Logger logger = Logger.getLogger(ProbeController.class);
	 
	@Autowired
	private ProbeService probeService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@Autowired
	private ProbeSetService probeSetService;
		
	@Autowired
	private GeneService geneService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ProbeResponse> createProbe(@Valid @RequestBody ProbeRequest probeRequest)
			throws DatabaseException
	{
		return _createProbe(probeRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProbeResponse> createProbeFromForm(@Valid @ModelAttribute ProbeRequest probeRequest)
			throws DatabaseException
	{
		return _createProbe(probeRequest);
	}

	private ResponseEntity<ProbeResponse> _createProbe(ProbeRequest probeRequest) throws DatabaseException
	{
		Probe probe = probeService.create(probeRequest.toProbe());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/probe/" + probe.getId());
		return new ResponseEntity<ProbeResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ProbeResponse retrieveProbe(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbe(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ProbeResponse retrieveProbeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbe(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ProbeResponse _retrieveProbe(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Probe probe = probeService.read(id);
		if (probe == null) throw new EntityNotFoundException("Probe " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ProbeResponse(probe, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveProbeXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProbeXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveProbeXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Probe probe = probeService.read(id);
		if (probe == null) throw new EntityNotFoundException("Probe " + id.toString() + " not found");
		Integer chromosomeId = probe.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/probeSet", method = RequestMethod.GET)
	public String retrieveProbeXrefProbeSet(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeXrefProbeSet(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/probeSet", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProbeXrefProbeSetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeXrefProbeSet(id, "json", expandFields);
	}
	
	private String _retrieveProbeXrefProbeSet(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Probe probe = probeService.read(id);
		if (probe == null) throw new EntityNotFoundException("Probe " + id.toString() + " not found");
		Integer probeSetId = probe.getProbeSet_Id();
		String redirectUri = "redirect:/api/v1/probeset/" + probeSetId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET)
	public String retrieveProbeXrefReportsFor(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeXrefReportsFor(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProbeXrefReportsForJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeXrefReportsFor(id, "json", expandFields);
	}
	
	private String _retrieveProbeXrefReportsFor(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Probe probe = probeService.read(id);
		if (probe == null) throw new EntityNotFoundException("Probe " + id.toString() + " not found");
		Integer geneId = probe.getReportsFor_Id();
		String redirectUri = "redirect:/api/v1/gene/" + geneId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateProbe(@PathVariable Integer id, @Valid @RequestBody ProbeRequest probeRequest)
			throws DatabaseException
	{
		_updateProbe(id, probeRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProbeResponse> updateProbeFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ProbeRequest probeRequest) throws DatabaseException
	{
		return _createProbe(probeRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProbePost(@PathVariable Integer id, @Valid @RequestBody ProbeRequest probeRequest)
			throws DatabaseException
	{
		_updateProbe(id, probeRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProbeFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ProbeRequest probeRequest)
			throws DatabaseException
	{
		_updateProbe(id, probeRequest);
	}

	private void _updateProbe(Integer id, ProbeRequest probeRequest) throws DatabaseException
	{
		Probe probe = probeRequest.toProbe();
		probe.setId(id);
		probeService.update(probe);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProbe(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProbe(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProbePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProbe(id);
	}

	private void _deleteProbe(Integer id) throws DatabaseException
	{
		boolean isDeleted = probeService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Probe " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ProbeResponse> retrieveProbeCollection(@Valid EntityCollectionRequest probeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeCollection(probeCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProbeResponse> retrieveProbeCollectionJson(@Valid EntityCollectionRequest probeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeCollection(probeCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ProbeResponse> retrieveProbeCollectionPost(@Valid @RequestBody EntityCollectionRequest probeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeCollection(probeCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProbeResponse> retrieveProbeCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest probeCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeCollection(probeCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ProbeResponse> _retrieveProbeCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Probe> probePager = probeService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ProbeResponse>(probePager, Lists.newArrayList(Iterables.transform(
				probePager.getIterable(), new Function<Probe, ProbeResponse>()
				{
					@Override
					@Nullable
					public ProbeResponse apply(@Nullable Probe probe)
					{
						try
						{
							return probe != null ? new ProbeResponse(probe, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/probe");
	}

	private static class ProbeRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
		private Boolean mismatch = false;	
		private Integer probeSet;
		private Integer reportsFor;
	
		public Probe toProbe()
		{
			Probe probe = new Probe();
			probe.setIdentifier(identifier);
			probe.setName(name);
			probe.setDescription(description);
			probe.setChromosome_Id(chromosome);
			probe.setCM(cM);
			probe.setBpStart(bpStart);
			probe.setBpEnd(bpEnd);
			probe.setSeq(seq);
			probe.setSymbol(symbol);
			probe.setMismatch(mismatch);
			probe.setProbeSet_Id(probeSet);
			probe.setReportsFor_Id(reportsFor);
			return probe;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
		public void setMismatch(Boolean mismatch)
		{
			this.mismatch = mismatch;
		}
		
		public void setProbeSet(Integer probeSet)
		{
			this.probeSet = probeSet;
		}
		
		public void setReportsFor(Integer reportsFor)
		{
			this.reportsFor = reportsFor;
		}
		
	}

	static class ProbeResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
		private final Boolean mismatch;
		private final Object probeSet;
		private final Object reportsFor;
	
		public ProbeResponse(Probe probe, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/probe/" + probe.getId();
			this.identifier = probe.getIdentifier();
			this.name = probe.getName();
			this.description = probe.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = probe.getChromosome() == null ? null : new ChromosomeResponse(probe.getChromosome(), null);
			else this.chromosome = probe.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/probe/" + probe.getId() + "/chromosome");
			this.cM = probe.getCM();
			this.bpStart = probe.getBpStart();
			this.bpEnd = probe.getBpEnd();
			this.seq = probe.getSeq();
			this.symbol = probe.getSymbol();
			this.mismatch = probe.getMismatch();
			if (expandFields != null && expandFields.contains("probeSet")) this.probeSet = probe.getProbeSet() == null ? null : new ProbeSetResponse(probe.getProbeSet(), null);
			else this.probeSet = probe.getProbeSet() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/probe/" + probe.getId() + "/probeSet");
			if (expandFields != null && expandFields.contains("reportsFor")) this.reportsFor = probe.getReportsFor() == null ? null : new GeneResponse(probe.getReportsFor(), null);
			else this.reportsFor = probe.getReportsFor() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/probe/" + probe.getId() + "/reportsFor");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
		public Boolean getMismatch()
		{
			return mismatch;
		}
	
		public Object getProbeSet()
		{
			return probeSet;
		}
	
		public Object getReportsFor()
		{
			return reportsFor;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}