package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Protein;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ProteinService;
import org.molgenis.omx.xgap.Gene;
import org.molgenis.controller.GeneController.GeneResponse;
import org.molgenis.service.GeneService;
import org.molgenis.omx.xgap.Transcript;
import org.molgenis.controller.TranscriptController.TranscriptResponse;
import org.molgenis.service.TranscriptService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/protein")
public class ProteinController
{
	private static Logger logger = Logger.getLogger(ProteinController.class);
	 
	@Autowired
	private ProteinService proteinService;

	@Autowired
	private GeneService geneService;
		
	@Autowired
	private TranscriptService transcriptService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ProteinResponse> createProtein(@Valid @RequestBody ProteinRequest proteinRequest)
			throws DatabaseException
	{
		return _createProtein(proteinRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProteinResponse> createProteinFromForm(@Valid @ModelAttribute ProteinRequest proteinRequest)
			throws DatabaseException
	{
		return _createProtein(proteinRequest);
	}

	private ResponseEntity<ProteinResponse> _createProtein(ProteinRequest proteinRequest) throws DatabaseException
	{
		Protein protein = proteinService.create(proteinRequest.toProtein());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/protein/" + protein.getId());
		return new ResponseEntity<ProteinResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ProteinResponse retrieveProtein(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtein(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ProteinResponse retrieveProteinJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtein(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ProteinResponse _retrieveProtein(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Protein protein = proteinService.read(id);
		if (protein == null) throw new EntityNotFoundException("Protein " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ProteinResponse(protein, expandFields);
	}
			
	@RequestMapping(value = "/{id}/gene", method = RequestMethod.GET)
	public String retrieveProteinXrefGene(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinXrefGene(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/gene", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProteinXrefGeneJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinXrefGene(id, "json", expandFields);
	}
	
	private String _retrieveProteinXrefGene(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Protein protein = proteinService.read(id);
		if (protein == null) throw new EntityNotFoundException("Protein " + id.toString() + " not found");
		Integer geneId = protein.getGene_Id();
		String redirectUri = "redirect:/api/v1/gene/" + geneId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/transcript", method = RequestMethod.GET)
	public String retrieveProteinXrefTranscript(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinXrefTranscript(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/transcript", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProteinXrefTranscriptJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinXrefTranscript(id, "json", expandFields);
	}
	
	private String _retrieveProteinXrefTranscript(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Protein protein = proteinService.read(id);
		if (protein == null) throw new EntityNotFoundException("Protein " + id.toString() + " not found");
		Integer transcriptId = protein.getTranscript_Id();
		String redirectUri = "redirect:/api/v1/transcript/" + transcriptId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateProtein(@PathVariable Integer id, @Valid @RequestBody ProteinRequest proteinRequest)
			throws DatabaseException
	{
		_updateProtein(id, proteinRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProteinResponse> updateProteinFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ProteinRequest proteinRequest) throws DatabaseException
	{
		return _createProtein(proteinRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProteinPost(@PathVariable Integer id, @Valid @RequestBody ProteinRequest proteinRequest)
			throws DatabaseException
	{
		_updateProtein(id, proteinRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProteinFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ProteinRequest proteinRequest)
			throws DatabaseException
	{
		_updateProtein(id, proteinRequest);
	}

	private void _updateProtein(Integer id, ProteinRequest proteinRequest) throws DatabaseException
	{
		Protein protein = proteinRequest.toProtein();
		protein.setId(id);
		proteinService.update(protein);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProtein(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProtein(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProteinPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProtein(id);
	}

	private void _deleteProtein(Integer id) throws DatabaseException
	{
		boolean isDeleted = proteinService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Protein " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ProteinResponse> retrieveProteinCollection(@Valid EntityCollectionRequest proteinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinCollection(proteinCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProteinResponse> retrieveProteinCollectionJson(@Valid EntityCollectionRequest proteinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinCollection(proteinCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ProteinResponse> retrieveProteinCollectionPost(@Valid @RequestBody EntityCollectionRequest proteinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinCollection(proteinCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProteinResponse> retrieveProteinCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest proteinCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProteinCollection(proteinCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ProteinResponse> _retrieveProteinCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Protein> proteinPager = proteinService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ProteinResponse>(proteinPager, Lists.newArrayList(Iterables.transform(
				proteinPager.getIterable(), new Function<Protein, ProteinResponse>()
				{
					@Override
					@Nullable
					public ProteinResponse apply(@Nullable Protein protein)
					{
						try
						{
							return protein != null ? new ProteinResponse(protein, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/protein");
	}

	private static class ProteinRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer gene;
		private Integer transcript;
		private String aminoSequence;
		private Double mass;
	
		public Protein toProtein()
		{
			Protein protein = new Protein();
			protein.setIdentifier(identifier);
			protein.setName(name);
			protein.setDescription(description);
			protein.setGene_Id(gene);
			protein.setTranscript_Id(transcript);
			protein.setAminoSequence(aminoSequence);
			protein.setMass(mass);
			return protein;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setGene(Integer gene)
		{
			this.gene = gene;
		}
		
		public void setTranscript(Integer transcript)
		{
			this.transcript = transcript;
		}
		
		public void setAminoSequence(String aminoSequence)
		{
			this.aminoSequence = aminoSequence;
		}
		
		public void setMass(Double mass)
		{
			this.mass = mass;
		}
		
	}

	static class ProteinResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object gene;
		private final Object transcript;
		private final String aminoSequence;
		private final Double mass;
	
		public ProteinResponse(Protein protein, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/protein/" + protein.getId();
			this.identifier = protein.getIdentifier();
			this.name = protein.getName();
			this.description = protein.getDescription();
			if (expandFields != null && expandFields.contains("gene")) this.gene = protein.getGene() == null ? null : new GeneResponse(protein.getGene(), null);
			else this.gene = protein.getGene() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/protein/" + protein.getId() + "/gene");
			if (expandFields != null && expandFields.contains("transcript")) this.transcript = protein.getTranscript() == null ? null : new TranscriptResponse(protein.getTranscript(), null);
			else this.transcript = protein.getTranscript() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/protein/" + protein.getId() + "/transcript");
			this.aminoSequence = protein.getAminoSequence();
			this.mass = protein.getMass();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getGene()
		{
			return gene;
		}
	
		public Object getTranscript()
		{
			return transcript;
		}
	
		public String getAminoSequence()
		{
			return aminoSequence;
		}
	
		public Double getMass()
		{
			return mass;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}