package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.workflow.ProtocolFlow;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ProtocolFlowService;
import org.molgenis.omx.observ.ObservableFeature;
import org.molgenis.controller.ObservableFeatureController.ObservableFeatureResponse;
import org.molgenis.service.ObservableFeatureService;
import org.molgenis.omx.observ.Protocol;
import org.molgenis.controller.ProtocolController.ProtocolResponse;
import org.molgenis.service.ProtocolService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/protocolflow")
public class ProtocolFlowController
{
	private static Logger logger = Logger.getLogger(ProtocolFlowController.class);
	 
	@Autowired
	private ProtocolFlowService protocolFlowService;

	@Autowired
	private ObservableFeatureService observableFeatureService;
		
	@Autowired
	private ProtocolService protocolService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ProtocolFlowResponse> createProtocolFlow(@Valid @RequestBody ProtocolFlowRequest protocolFlowRequest)
			throws DatabaseException
	{
		return _createProtocolFlow(protocolFlowRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProtocolFlowResponse> createProtocolFlowFromForm(@Valid @ModelAttribute ProtocolFlowRequest protocolFlowRequest)
			throws DatabaseException
	{
		return _createProtocolFlow(protocolFlowRequest);
	}

	private ResponseEntity<ProtocolFlowResponse> _createProtocolFlow(ProtocolFlowRequest protocolFlowRequest) throws DatabaseException
	{
		ProtocolFlow protocolFlow = protocolFlowService.create(protocolFlowRequest.toProtocolFlow());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/protocolflow/" + protocolFlow.getId());
		return new ResponseEntity<ProtocolFlowResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ProtocolFlowResponse retrieveProtocolFlow(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlow(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ProtocolFlowResponse retrieveProtocolFlowJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlow(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ProtocolFlowResponse _retrieveProtocolFlow(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		ProtocolFlow protocolFlow = protocolFlowService.read(id);
		if (protocolFlow == null) throw new EntityNotFoundException("ProtocolFlow " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ProtocolFlowResponse(protocolFlow, expandFields);
	}
			
	@RequestMapping(value = "/{id}/inputFeature", method = RequestMethod.GET)
	public String retrieveProtocolFlowXrefInputFeature(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefInputFeature(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/inputFeature", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProtocolFlowXrefInputFeatureJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefInputFeature(id, "json", expandFields);
	}
	
	private String _retrieveProtocolFlowXrefInputFeature(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ProtocolFlow protocolFlow = protocolFlowService.read(id);
		if (protocolFlow == null) throw new EntityNotFoundException("ProtocolFlow " + id.toString() + " not found");
		Integer observableFeatureId = protocolFlow.getInputFeature_Id();
		String redirectUri = "redirect:/api/v1/observablefeature/" + observableFeatureId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/outputFeature", method = RequestMethod.GET)
	public String retrieveProtocolFlowXrefOutputFeature(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefOutputFeature(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/outputFeature", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProtocolFlowXrefOutputFeatureJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefOutputFeature(id, "json", expandFields);
	}
	
	private String _retrieveProtocolFlowXrefOutputFeature(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ProtocolFlow protocolFlow = protocolFlowService.read(id);
		if (protocolFlow == null) throw new EntityNotFoundException("ProtocolFlow " + id.toString() + " not found");
		Integer observableFeatureId = protocolFlow.getOutputFeature_Id();
		String redirectUri = "redirect:/api/v1/observablefeature/" + observableFeatureId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/source", method = RequestMethod.GET)
	public String retrieveProtocolFlowXrefSource(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefSource(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/source", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProtocolFlowXrefSourceJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefSource(id, "json", expandFields);
	}
	
	private String _retrieveProtocolFlowXrefSource(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ProtocolFlow protocolFlow = protocolFlowService.read(id);
		if (protocolFlow == null) throw new EntityNotFoundException("ProtocolFlow " + id.toString() + " not found");
		Integer protocolId = protocolFlow.getSource_Id();
		String redirectUri = "redirect:/api/v1/protocol/" + protocolId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/destination", method = RequestMethod.GET)
	public String retrieveProtocolFlowXrefDestination(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefDestination(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/destination", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProtocolFlowXrefDestinationJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowXrefDestination(id, "json", expandFields);
	}
	
	private String _retrieveProtocolFlowXrefDestination(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ProtocolFlow protocolFlow = protocolFlowService.read(id);
		if (protocolFlow == null) throw new EntityNotFoundException("ProtocolFlow " + id.toString() + " not found");
		Integer protocolId = protocolFlow.getDestination_Id();
		String redirectUri = "redirect:/api/v1/protocol/" + protocolId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateProtocolFlow(@PathVariable Integer id, @Valid @RequestBody ProtocolFlowRequest protocolFlowRequest)
			throws DatabaseException
	{
		_updateProtocolFlow(id, protocolFlowRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProtocolFlowResponse> updateProtocolFlowFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ProtocolFlowRequest protocolFlowRequest) throws DatabaseException
	{
		return _createProtocolFlow(protocolFlowRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProtocolFlowPost(@PathVariable Integer id, @Valid @RequestBody ProtocolFlowRequest protocolFlowRequest)
			throws DatabaseException
	{
		_updateProtocolFlow(id, protocolFlowRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProtocolFlowFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ProtocolFlowRequest protocolFlowRequest)
			throws DatabaseException
	{
		_updateProtocolFlow(id, protocolFlowRequest);
	}

	private void _updateProtocolFlow(Integer id, ProtocolFlowRequest protocolFlowRequest) throws DatabaseException
	{
		ProtocolFlow protocolFlow = protocolFlowRequest.toProtocolFlow();
		protocolFlow.setId(id);
		protocolFlowService.update(protocolFlow);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProtocolFlow(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProtocolFlow(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProtocolFlowPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProtocolFlow(id);
	}

	private void _deleteProtocolFlow(Integer id) throws DatabaseException
	{
		boolean isDeleted = protocolFlowService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("ProtocolFlow " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ProtocolFlowResponse> retrieveProtocolFlowCollection(@Valid EntityCollectionRequest protocolFlowCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowCollection(protocolFlowCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProtocolFlowResponse> retrieveProtocolFlowCollectionJson(@Valid EntityCollectionRequest protocolFlowCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowCollection(protocolFlowCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ProtocolFlowResponse> retrieveProtocolFlowCollectionPost(@Valid @RequestBody EntityCollectionRequest protocolFlowCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowCollection(protocolFlowCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProtocolFlowResponse> retrieveProtocolFlowCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest protocolFlowCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProtocolFlowCollection(protocolFlowCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ProtocolFlowResponse> _retrieveProtocolFlowCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<ProtocolFlow> protocolFlowPager = protocolFlowService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ProtocolFlowResponse>(protocolFlowPager, Lists.newArrayList(Iterables.transform(
				protocolFlowPager.getIterable(), new Function<ProtocolFlow, ProtocolFlowResponse>()
				{
					@Override
					@Nullable
					public ProtocolFlowResponse apply(@Nullable ProtocolFlow protocolFlow)
					{
						try
						{
							return protocolFlow != null ? new ProtocolFlowResponse(protocolFlow, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/protocolflow");
	}

	private static class ProtocolFlowRequest
	{
		private Integer inputFeature;
		private Integer outputFeature;
		private Integer source;
		private Integer destination;
	
		public ProtocolFlow toProtocolFlow()
		{
			ProtocolFlow protocolFlow = new ProtocolFlow();
			protocolFlow.setInputFeature_Id(inputFeature);
			protocolFlow.setOutputFeature_Id(outputFeature);
			protocolFlow.setSource_Id(source);
			protocolFlow.setDestination_Id(destination);
			return protocolFlow;
		}
		
		public void setInputFeature(Integer inputFeature)
		{
			this.inputFeature = inputFeature;
		}
		
		public void setOutputFeature(Integer outputFeature)
		{
			this.outputFeature = outputFeature;
		}
		
		public void setSource(Integer source)
		{
			this.source = source;
		}
		
		public void setDestination(Integer destination)
		{
			this.destination = destination;
		}
		
	}

	static class ProtocolFlowResponse
	{
		private final String href;
		private final Object inputFeature;
		private final Object outputFeature;
		private final Object source;
		private final Object destination;
	
		public ProtocolFlowResponse(ProtocolFlow protocolFlow, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/protocolflow/" + protocolFlow.getId();
			if (expandFields != null && expandFields.contains("inputFeature")) this.inputFeature = new ObservableFeatureResponse(protocolFlow.getInputFeature(), null);
			else this.inputFeature = java.util.Collections.singletonMap("href", "/api/v1/protocolflow/" + protocolFlow.getId() + "/inputFeature");
			if (expandFields != null && expandFields.contains("outputFeature")) this.outputFeature = new ObservableFeatureResponse(protocolFlow.getOutputFeature(), null);
			else this.outputFeature = java.util.Collections.singletonMap("href", "/api/v1/protocolflow/" + protocolFlow.getId() + "/outputFeature");
			if (expandFields != null && expandFields.contains("source")) this.source = new ProtocolResponse(protocolFlow.getSource(), null);
			else this.source = java.util.Collections.singletonMap("href", "/api/v1/protocolflow/" + protocolFlow.getId() + "/source");
			if (expandFields != null && expandFields.contains("destination")) this.destination = new ProtocolResponse(protocolFlow.getDestination(), null);
			else this.destination = java.util.Collections.singletonMap("href", "/api/v1/protocolflow/" + protocolFlow.getId() + "/destination");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public Object getInputFeature()
		{
			return inputFeature;
		}
	
		public Object getOutputFeature()
		{
			return outputFeature;
		}
	
		public Object getSource()
		{
			return source;
		}
	
		public Object getDestination()
		{
			return destination;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}