package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.SNP;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.SNPService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.omx.xgap.Marker;
import org.molgenis.controller.MarkerController.MarkerResponse;
import org.molgenis.service.MarkerService;
import org.molgenis.omx.xgap.Polymorphism;
import org.molgenis.controller.PolymorphismController.PolymorphismResponse;
import org.molgenis.service.PolymorphismService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/snp")
public class SNPController
{
	private static Logger logger = Logger.getLogger(SNPController.class);
	 
	@Autowired
	private SNPService sNPService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@Autowired
	private MarkerService markerService;
		
	@Autowired
	private PolymorphismService polymorphismService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<SNPResponse> createSNP(@Valid @RequestBody SNPRequest sNPRequest)
			throws DatabaseException
	{
		return _createSNP(sNPRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SNPResponse> createSNPFromForm(@Valid @ModelAttribute SNPRequest sNPRequest)
			throws DatabaseException
	{
		return _createSNP(sNPRequest);
	}

	private ResponseEntity<SNPResponse> _createSNP(SNPRequest sNPRequest) throws DatabaseException
	{
		SNP sNP = sNPService.create(sNPRequest.toSNP());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/snp/" + sNP.getId());
		return new ResponseEntity<SNPResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public SNPResponse retrieveSNP(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNP(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public SNPResponse retrieveSNPJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNP(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private SNPResponse _retrieveSNP(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		SNP sNP = sNPService.read(id);
		if (sNP == null) throw new EntityNotFoundException("SNP " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new SNPResponse(sNP, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveSNPXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNPXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSNPXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNPXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveSNPXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		SNP sNP = sNPService.read(id);
		if (sNP == null) throw new EntityNotFoundException("SNP " + id.toString() + " not found");
		Integer chromosomeId = sNP.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	

	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveSNPMrefReportsFor(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		SNP sNP = sNPService.read(id);
		if (sNP == null) throw new EntityNotFoundException("SNP " + id.toString() + " not found");
		return _retrieveSNPMrefReportsFor(sNP, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MarkerResponse> retrieveSNPMrefReportsForJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		SNP sNP = sNPService.read(id);
		if (sNP == null) throw new EntityNotFoundException("SNP " + id.toString() + " not found");
		return _retrieveSNPMrefReportsFor(sNP, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<MarkerResponse> _retrieveSNPMrefReportsFor(SNP sNP, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<Marker> markerCollection = sNP.getReportsFor();
		
		int total = markerCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		markerCollection = markerCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<Marker> markerPager = new EntityPager<Marker>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, markerCollection);
		
		return new EntityCollectionResponse<MarkerResponse>(markerPager, Lists.newArrayList(Iterables.transform(markerCollection,
				new Function<Marker, MarkerResponse>()
				{
					@Override
					@Nullable
					public MarkerResponse apply(@Nullable Marker marker)
					{
						try
						{
							return marker != null ? new MarkerResponse(marker, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/snp/" + sNP.getId() + "/reportsFor");
	}
	@RequestMapping(value = "/{id}/polymorphism", method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<PolymorphismResponse> retrieveSNPMrefPolymorphism(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		SNP sNP = sNPService.read(id);
		if (sNP == null) throw new EntityNotFoundException("SNP " + id.toString() + " not found");
		return _retrieveSNPMrefPolymorphism(sNP, entityCollectionRequest, expandFields);
	}
	
	@RequestMapping(value = "/{id}/polymorphism", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<PolymorphismResponse> retrieveSNPMrefPolymorphismJson(@PathVariable Integer id, @Valid EntityCollectionRequest entityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		SNP sNP = sNPService.read(id);
		if (sNP == null) throw new EntityNotFoundException("SNP " + id.toString() + " not found");
		return _retrieveSNPMrefPolymorphism(sNP, entityCollectionRequest, expandFields);
	}
	
	private static EntityCollectionResponse<PolymorphismResponse> _retrieveSNPMrefPolymorphism(SNP sNP, EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		java.util.List<Polymorphism> polymorphismCollection = sNP.getPolymorphism();
		
		int total = polymorphismCollection.size();
		int toIndex = entityCollectionRequest.getStart() + entityCollectionRequest.getNum();
		polymorphismCollection = polymorphismCollection.subList(entityCollectionRequest.getStart(),
				toIndex > total ? total : toIndex);
		
		
		EntityPager<Polymorphism> polymorphismPager = new EntityPager<Polymorphism>(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), total, polymorphismCollection);
		
		return new EntityCollectionResponse<PolymorphismResponse>(polymorphismPager, Lists.newArrayList(Iterables.transform(polymorphismCollection,
				new Function<Polymorphism, PolymorphismResponse>()
				{
					@Override
					@Nullable
					public PolymorphismResponse apply(@Nullable Polymorphism polymorphism)
					{
						try
						{
							return polymorphism != null ? new PolymorphismResponse(polymorphism, expandFields) : null;
						}
						catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/snp/" + sNP.getId() + "/polymorphism");
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateSNP(@PathVariable Integer id, @Valid @RequestBody SNPRequest sNPRequest)
			throws DatabaseException
	{
		_updateSNP(id, sNPRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SNPResponse> updateSNPFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute SNPRequest sNPRequest) throws DatabaseException
	{
		return _createSNP(sNPRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSNPPost(@PathVariable Integer id, @Valid @RequestBody SNPRequest sNPRequest)
			throws DatabaseException
	{
		_updateSNP(id, sNPRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSNPFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute SNPRequest sNPRequest)
			throws DatabaseException
	{
		_updateSNP(id, sNPRequest);
	}

	private void _updateSNP(Integer id, SNPRequest sNPRequest) throws DatabaseException
	{
		SNP sNP = sNPRequest.toSNP();
		sNP.setId(id);
		sNPService.update(sNP);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSNP(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSNP(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSNPPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSNP(id);
	}

	private void _deleteSNP(Integer id) throws DatabaseException
	{
		boolean isDeleted = sNPService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("SNP " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<SNPResponse> retrieveSNPCollection(@Valid EntityCollectionRequest sNPCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNPCollection(sNPCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SNPResponse> retrieveSNPCollectionJson(@Valid EntityCollectionRequest sNPCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNPCollection(sNPCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<SNPResponse> retrieveSNPCollectionPost(@Valid @RequestBody EntityCollectionRequest sNPCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNPCollection(sNPCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SNPResponse> retrieveSNPCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest sNPCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSNPCollection(sNPCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<SNPResponse> _retrieveSNPCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<SNP> sNPPager = sNPService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<SNPResponse>(sNPPager, Lists.newArrayList(Iterables.transform(
				sNPPager.getIterable(), new Function<SNP, SNPResponse>()
				{
					@Override
					@Nullable
					public SNPResponse apply(@Nullable SNP sNP)
					{
						try
						{
							return sNP != null ? new SNPResponse(sNP, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/snp");
	}

	private static class SNPRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
		private java.util.List<Integer> reportsFor;
		private String status;
		private java.util.List<Integer> polymorphism;
	
		public SNP toSNP()
		{
			SNP sNP = new SNP();
			sNP.setIdentifier(identifier);
			sNP.setName(name);
			sNP.setDescription(description);
			sNP.setChromosome_Id(chromosome);
			sNP.setCM(cM);
			sNP.setBpStart(bpStart);
			sNP.setBpEnd(bpEnd);
			sNP.setSeq(seq);
			sNP.setSymbol(symbol);
			sNP.setReportsFor_Id(reportsFor);
			sNP.setStatus(status);
			sNP.setPolymorphism_Id(polymorphism);
			return sNP;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
		public void setReportsFor(java.util.List<Integer> reportsFor)
		{
			this.reportsFor = reportsFor;
		}
		
		public void setStatus(String status)
		{
			this.status = status;
		}
		
		public void setPolymorphism(java.util.List<Integer> polymorphism)
		{
			this.polymorphism = polymorphism;
		}
		
	}

	static class SNPResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
		private final Object reportsFor;
		private final String status;
		private final Object polymorphism;
	
		public SNPResponse(SNP sNP, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/snp/" + sNP.getId();
			this.identifier = sNP.getIdentifier();
			this.name = sNP.getName();
			this.description = sNP.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = sNP.getChromosome() == null ? null : new ChromosomeResponse(sNP.getChromosome(), null);
			else this.chromosome = sNP.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/snp/" + sNP.getId() + "/chromosome");
			this.cM = sNP.getCM();
			this.bpStart = sNP.getBpStart();
			this.bpEnd = sNP.getBpEnd();
			this.seq = sNP.getSeq();
			this.symbol = sNP.getSymbol();
			java.util.List<Marker> ReportsForCollection = sNP.getReportsFor();
			if (expandFields != null && expandFields.contains("reportsFor")) this.reportsFor = ReportsForCollection == null ? null : _retrieveSNPMrefReportsFor(sNP, new EntityCollectionRequest());
			else this.reportsFor = ReportsForCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/snp/" + sNP.getId() + "/reportsFor");
			this.status = sNP.getStatus();
			java.util.List<Polymorphism> PolymorphismCollection = sNP.getPolymorphism();
			if (expandFields != null && expandFields.contains("polymorphism")) this.polymorphism = PolymorphismCollection == null ? null : _retrieveSNPMrefPolymorphism(sNP, new EntityCollectionRequest());
			else this.polymorphism = PolymorphismCollection == null ? null : java.util.Collections.singletonMap("href", "/api/v1/snp/" + sNP.getId() + "/polymorphism");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
		public Object getReportsFor()
		{
			return reportsFor;
		}
	
		public String getStatus()
		{
			return status;
		}
	
		public Object getPolymorphism()
		{
			return polymorphism;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}