package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Sample;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.SampleService;
import org.molgenis.omx.observ.target.Individual;
import org.molgenis.controller.IndividualController.IndividualResponse;
import org.molgenis.service.IndividualService;
import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.controller.OntologyTermController.OntologyTermResponse;
import org.molgenis.service.OntologyTermService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/sample")
public class SampleController
{
	private static Logger logger = Logger.getLogger(SampleController.class);
	 
	@Autowired
	private SampleService sampleService;

	@Autowired
	private IndividualService individualService;
		
	@Autowired
	private OntologyTermService ontologyTermService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<SampleResponse> createSample(@Valid @RequestBody SampleRequest sampleRequest)
			throws DatabaseException
	{
		return _createSample(sampleRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SampleResponse> createSampleFromForm(@Valid @ModelAttribute SampleRequest sampleRequest)
			throws DatabaseException
	{
		return _createSample(sampleRequest);
	}

	private ResponseEntity<SampleResponse> _createSample(SampleRequest sampleRequest) throws DatabaseException
	{
		Sample sample = sampleService.create(sampleRequest.toSample());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/sample/" + sample.getId());
		return new ResponseEntity<SampleResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public SampleResponse retrieveSample(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSample(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public SampleResponse retrieveSampleJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSample(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private SampleResponse _retrieveSample(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new SampleResponse(sample, expandFields);
	}
			
	@RequestMapping(value = "/{id}/individual", method = RequestMethod.GET)
	public String retrieveSampleXrefIndividual(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefIndividual(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/individual", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSampleXrefIndividualJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefIndividual(id, "json", expandFields);
	}
	
	private String _retrieveSampleXrefIndividual(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		Integer individualId = sample.getIndividual_Id();
		String redirectUri = "redirect:/api/v1/individual/" + individualId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/tissue", method = RequestMethod.GET)
	public String retrieveSampleXrefTissue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefTissue(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/tissue", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSampleXrefTissueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleXrefTissue(id, "json", expandFields);
	}
	
	private String _retrieveSampleXrefTissue(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Sample sample = sampleService.read(id);
		if (sample == null) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
		Integer ontologyTermId = sample.getTissue_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateSample(@PathVariable Integer id, @Valid @RequestBody SampleRequest sampleRequest)
			throws DatabaseException
	{
		_updateSample(id, sampleRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SampleResponse> updateSampleFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute SampleRequest sampleRequest) throws DatabaseException
	{
		return _createSample(sampleRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSamplePost(@PathVariable Integer id, @Valid @RequestBody SampleRequest sampleRequest)
			throws DatabaseException
	{
		_updateSample(id, sampleRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSampleFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute SampleRequest sampleRequest)
			throws DatabaseException
	{
		_updateSample(id, sampleRequest);
	}

	private void _updateSample(Integer id, SampleRequest sampleRequest) throws DatabaseException
	{
		Sample sample = sampleRequest.toSample();
		sample.setId(id);
		sampleService.update(sample);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSample(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSample(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSamplePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSample(id);
	}

	private void _deleteSample(Integer id) throws DatabaseException
	{
		boolean isDeleted = sampleService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Sample " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollection(@Valid EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollectionJson(@Valid EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollectionPost(@Valid @RequestBody EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SampleResponse> retrieveSampleCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest sampleCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSampleCollection(sampleCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<SampleResponse> _retrieveSampleCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Sample> samplePager = sampleService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<SampleResponse>(samplePager, Lists.newArrayList(Iterables.transform(
				samplePager.getIterable(), new Function<Sample, SampleResponse>()
				{
					@Override
					@Nullable
					public SampleResponse apply(@Nullable Sample sample)
					{
						try
						{
							return sample != null ? new SampleResponse(sample, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/sample");
	}

	private static class SampleRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer individual;
		private Integer tissue;
	
		public Sample toSample()
		{
			Sample sample = new Sample();
			sample.setIdentifier(identifier);
			sample.setName(name);
			sample.setDescription(description);
			sample.setIndividual_Id(individual);
			sample.setTissue_Id(tissue);
			return sample;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setIndividual(Integer individual)
		{
			this.individual = individual;
		}
		
		public void setTissue(Integer tissue)
		{
			this.tissue = tissue;
		}
		
	}

	static class SampleResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object individual;
		private final Object tissue;
	
		public SampleResponse(Sample sample, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/sample/" + sample.getId();
			this.identifier = sample.getIdentifier();
			this.name = sample.getName();
			this.description = sample.getDescription();
			if (expandFields != null && expandFields.contains("individual")) this.individual = sample.getIndividual() == null ? null : new IndividualResponse(sample.getIndividual(), null);
			else this.individual = sample.getIndividual() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/sample/" + sample.getId() + "/individual");
			if (expandFields != null && expandFields.contains("tissue")) this.tissue = sample.getTissue() == null ? null : new OntologyTermResponse(sample.getTissue(), null);
			else this.tissue = sample.getTissue() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/sample/" + sample.getId() + "/tissue");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getIndividual()
		{
			return individual;
		}
	
		public Object getTissue()
		{
			return tissue;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}