package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Spot;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.SpotService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.omx.xgap.ProbeSet;
import org.molgenis.controller.ProbeSetController.ProbeSetResponse;
import org.molgenis.service.ProbeSetService;
import org.molgenis.omx.xgap.Gene;
import org.molgenis.controller.GeneController.GeneResponse;
import org.molgenis.service.GeneService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/spot")
public class SpotController
{
	private static Logger logger = Logger.getLogger(SpotController.class);
	 
	@Autowired
	private SpotService spotService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@Autowired
	private ProbeSetService probeSetService;
		
	@Autowired
	private GeneService geneService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<SpotResponse> createSpot(@Valid @RequestBody SpotRequest spotRequest)
			throws DatabaseException
	{
		return _createSpot(spotRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SpotResponse> createSpotFromForm(@Valid @ModelAttribute SpotRequest spotRequest)
			throws DatabaseException
	{
		return _createSpot(spotRequest);
	}

	private ResponseEntity<SpotResponse> _createSpot(SpotRequest spotRequest) throws DatabaseException
	{
		Spot spot = spotService.create(spotRequest.toSpot());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/spot/" + spot.getId());
		return new ResponseEntity<SpotResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public SpotResponse retrieveSpot(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpot(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public SpotResponse retrieveSpotJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpot(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private SpotResponse _retrieveSpot(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Spot spot = spotService.read(id);
		if (spot == null) throw new EntityNotFoundException("Spot " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new SpotResponse(spot, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveSpotXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSpotXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveSpotXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Spot spot = spotService.read(id);
		if (spot == null) throw new EntityNotFoundException("Spot " + id.toString() + " not found");
		Integer chromosomeId = spot.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/probeSet", method = RequestMethod.GET)
	public String retrieveSpotXrefProbeSet(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotXrefProbeSet(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/probeSet", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSpotXrefProbeSetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotXrefProbeSet(id, "json", expandFields);
	}
	
	private String _retrieveSpotXrefProbeSet(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Spot spot = spotService.read(id);
		if (spot == null) throw new EntityNotFoundException("Spot " + id.toString() + " not found");
		Integer probeSetId = spot.getProbeSet_Id();
		String redirectUri = "redirect:/api/v1/probeset/" + probeSetId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET)
	public String retrieveSpotXrefReportsFor(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotXrefReportsFor(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/reportsFor", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveSpotXrefReportsForJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotXrefReportsFor(id, "json", expandFields);
	}
	
	private String _retrieveSpotXrefReportsFor(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Spot spot = spotService.read(id);
		if (spot == null) throw new EntityNotFoundException("Spot " + id.toString() + " not found");
		Integer geneId = spot.getReportsFor_Id();
		String redirectUri = "redirect:/api/v1/gene/" + geneId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateSpot(@PathVariable Integer id, @Valid @RequestBody SpotRequest spotRequest)
			throws DatabaseException
	{
		_updateSpot(id, spotRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<SpotResponse> updateSpotFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute SpotRequest spotRequest) throws DatabaseException
	{
		return _createSpot(spotRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSpotPost(@PathVariable Integer id, @Valid @RequestBody SpotRequest spotRequest)
			throws DatabaseException
	{
		_updateSpot(id, spotRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateSpotFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute SpotRequest spotRequest)
			throws DatabaseException
	{
		_updateSpot(id, spotRequest);
	}

	private void _updateSpot(Integer id, SpotRequest spotRequest) throws DatabaseException
	{
		Spot spot = spotRequest.toSpot();
		spot.setId(id);
		spotService.update(spot);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSpot(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSpot(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteSpotPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteSpot(id);
	}

	private void _deleteSpot(Integer id) throws DatabaseException
	{
		boolean isDeleted = spotService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Spot " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<SpotResponse> retrieveSpotCollection(@Valid EntityCollectionRequest spotCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotCollection(spotCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SpotResponse> retrieveSpotCollectionJson(@Valid EntityCollectionRequest spotCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotCollection(spotCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<SpotResponse> retrieveSpotCollectionPost(@Valid @RequestBody EntityCollectionRequest spotCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotCollection(spotCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<SpotResponse> retrieveSpotCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest spotCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveSpotCollection(spotCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<SpotResponse> _retrieveSpotCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Spot> spotPager = spotService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<SpotResponse>(spotPager, Lists.newArrayList(Iterables.transform(
				spotPager.getIterable(), new Function<Spot, SpotResponse>()
				{
					@Override
					@Nullable
					public SpotResponse apply(@Nullable Spot spot)
					{
						try
						{
							return spot != null ? new SpotResponse(spot, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/spot");
	}

	private static class SpotRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
		private Boolean mismatch = false;	
		private Integer probeSet;
		private Integer reportsFor;
		private Integer x;
		private Integer y;
		private Integer gridX;
		private Integer gridY;
	
		public Spot toSpot()
		{
			Spot spot = new Spot();
			spot.setIdentifier(identifier);
			spot.setName(name);
			spot.setDescription(description);
			spot.setChromosome_Id(chromosome);
			spot.setCM(cM);
			spot.setBpStart(bpStart);
			spot.setBpEnd(bpEnd);
			spot.setSeq(seq);
			spot.setSymbol(symbol);
			spot.setMismatch(mismatch);
			spot.setProbeSet_Id(probeSet);
			spot.setReportsFor_Id(reportsFor);
			spot.setX(x);
			spot.setY(y);
			spot.setGridX(gridX);
			spot.setGridY(gridY);
			return spot;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
		public void setMismatch(Boolean mismatch)
		{
			this.mismatch = mismatch;
		}
		
		public void setProbeSet(Integer probeSet)
		{
			this.probeSet = probeSet;
		}
		
		public void setReportsFor(Integer reportsFor)
		{
			this.reportsFor = reportsFor;
		}
		
		public void setX(Integer x)
		{
			this.x = x;
		}
		
		public void setY(Integer y)
		{
			this.y = y;
		}
		
		public void setGridX(Integer gridX)
		{
			this.gridX = gridX;
		}
		
		public void setGridY(Integer gridY)
		{
			this.gridY = gridY;
		}
		
	}

	static class SpotResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
		private final Boolean mismatch;
		private final Object probeSet;
		private final Object reportsFor;
		private final Integer x;
		private final Integer y;
		private final Integer gridX;
		private final Integer gridY;
	
		public SpotResponse(Spot spot, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/spot/" + spot.getId();
			this.identifier = spot.getIdentifier();
			this.name = spot.getName();
			this.description = spot.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = spot.getChromosome() == null ? null : new ChromosomeResponse(spot.getChromosome(), null);
			else this.chromosome = spot.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/spot/" + spot.getId() + "/chromosome");
			this.cM = spot.getCM();
			this.bpStart = spot.getBpStart();
			this.bpEnd = spot.getBpEnd();
			this.seq = spot.getSeq();
			this.symbol = spot.getSymbol();
			this.mismatch = spot.getMismatch();
			if (expandFields != null && expandFields.contains("probeSet")) this.probeSet = spot.getProbeSet() == null ? null : new ProbeSetResponse(spot.getProbeSet(), null);
			else this.probeSet = spot.getProbeSet() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/spot/" + spot.getId() + "/probeSet");
			if (expandFields != null && expandFields.contains("reportsFor")) this.reportsFor = spot.getReportsFor() == null ? null : new GeneResponse(spot.getReportsFor(), null);
			else this.reportsFor = spot.getReportsFor() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/spot/" + spot.getId() + "/reportsFor");
			this.x = spot.getX();
			this.y = spot.getY();
			this.gridX = spot.getGridX();
			this.gridY = spot.getGridY();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
		public Boolean getMismatch()
		{
			return mismatch;
		}
	
		public Object getProbeSet()
		{
			return probeSet;
		}
	
		public Object getReportsFor()
		{
			return reportsFor;
		}
	
		public Integer getX()
		{
			return x;
		}
	
		public Integer getY()
		{
			return y;
		}
	
		public Integer getGridX()
		{
			return gridX;
		}
	
		public Integer getGridY()
		{
			return gridY;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}