package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Transcript;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.TranscriptService;
import org.molgenis.omx.xgap.Gene;
import org.molgenis.controller.GeneController.GeneResponse;
import org.molgenis.service.GeneService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/transcript")
public class TranscriptController
{
	private static Logger logger = Logger.getLogger(TranscriptController.class);
	 
	@Autowired
	private TranscriptService transcriptService;

	@Autowired
	private GeneService geneService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<TranscriptResponse> createTranscript(@Valid @RequestBody TranscriptRequest transcriptRequest)
			throws DatabaseException
	{
		return _createTranscript(transcriptRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<TranscriptResponse> createTranscriptFromForm(@Valid @ModelAttribute TranscriptRequest transcriptRequest)
			throws DatabaseException
	{
		return _createTranscript(transcriptRequest);
	}

	private ResponseEntity<TranscriptResponse> _createTranscript(TranscriptRequest transcriptRequest) throws DatabaseException
	{
		Transcript transcript = transcriptService.create(transcriptRequest.toTranscript());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/transcript/" + transcript.getId());
		return new ResponseEntity<TranscriptResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public TranscriptResponse retrieveTranscript(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscript(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public TranscriptResponse retrieveTranscriptJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscript(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private TranscriptResponse _retrieveTranscript(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Transcript transcript = transcriptService.read(id);
		if (transcript == null) throw new EntityNotFoundException("Transcript " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new TranscriptResponse(transcript, expandFields);
	}
			
	@RequestMapping(value = "/{id}/gene", method = RequestMethod.GET)
	public String retrieveTranscriptXrefGene(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscriptXrefGene(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/gene", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveTranscriptXrefGeneJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscriptXrefGene(id, "json", expandFields);
	}
	
	private String _retrieveTranscriptXrefGene(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Transcript transcript = transcriptService.read(id);
		if (transcript == null) throw new EntityNotFoundException("Transcript " + id.toString() + " not found");
		Integer geneId = transcript.getGene_Id();
		String redirectUri = "redirect:/api/v1/gene/" + geneId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateTranscript(@PathVariable Integer id, @Valid @RequestBody TranscriptRequest transcriptRequest)
			throws DatabaseException
	{
		_updateTranscript(id, transcriptRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<TranscriptResponse> updateTranscriptFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute TranscriptRequest transcriptRequest) throws DatabaseException
	{
		return _createTranscript(transcriptRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateTranscriptPost(@PathVariable Integer id, @Valid @RequestBody TranscriptRequest transcriptRequest)
			throws DatabaseException
	{
		_updateTranscript(id, transcriptRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateTranscriptFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute TranscriptRequest transcriptRequest)
			throws DatabaseException
	{
		_updateTranscript(id, transcriptRequest);
	}

	private void _updateTranscript(Integer id, TranscriptRequest transcriptRequest) throws DatabaseException
	{
		Transcript transcript = transcriptRequest.toTranscript();
		transcript.setId(id);
		transcriptService.update(transcript);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteTranscript(@PathVariable Integer id) throws DatabaseException
	{
		_deleteTranscript(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteTranscriptPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteTranscript(id);
	}

	private void _deleteTranscript(Integer id) throws DatabaseException
	{
		boolean isDeleted = transcriptService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Transcript " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<TranscriptResponse> retrieveTranscriptCollection(@Valid EntityCollectionRequest transcriptCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscriptCollection(transcriptCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<TranscriptResponse> retrieveTranscriptCollectionJson(@Valid EntityCollectionRequest transcriptCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscriptCollection(transcriptCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<TranscriptResponse> retrieveTranscriptCollectionPost(@Valid @RequestBody EntityCollectionRequest transcriptCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscriptCollection(transcriptCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<TranscriptResponse> retrieveTranscriptCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest transcriptCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveTranscriptCollection(transcriptCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<TranscriptResponse> _retrieveTranscriptCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Transcript> transcriptPager = transcriptService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<TranscriptResponse>(transcriptPager, Lists.newArrayList(Iterables.transform(
				transcriptPager.getIterable(), new Function<Transcript, TranscriptResponse>()
				{
					@Override
					@Nullable
					public TranscriptResponse apply(@Nullable Transcript transcript)
					{
						try
						{
							return transcript != null ? new TranscriptResponse(transcript, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/transcript");
	}

	private static class TranscriptRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer gene;
	
		public Transcript toTranscript()
		{
			Transcript transcript = new Transcript();
			transcript.setIdentifier(identifier);
			transcript.setName(name);
			transcript.setDescription(description);
			transcript.setGene_Id(gene);
			return transcript;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setGene(Integer gene)
		{
			this.gene = gene;
		}
		
	}

	static class TranscriptResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object gene;
	
		public TranscriptResponse(Transcript transcript, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/transcript/" + transcript.getId();
			this.identifier = transcript.getIdentifier();
			this.name = transcript.getName();
			this.description = transcript.getDescription();
			if (expandFields != null && expandFields.contains("gene")) this.gene = transcript.getGene() == null ? null : new GeneResponse(transcript.getGene(), null);
			else this.gene = transcript.getGene() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/transcript/" + transcript.getId() + "/gene");
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getGene()
		{
			return gene;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}