package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.Category;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.CategoryService;
import org.molgenis.omx.observ.ObservableFeature;
import org.molgenis.controller.ObservableFeatureController.ObservableFeatureResponse;
import org.molgenis.service.ObservableFeatureService;
import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.controller.OntologyTermController.OntologyTermResponse;
import org.molgenis.service.OntologyTermService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/category")
public class CategoryController
{
	private static Logger logger = Logger.getLogger(CategoryController.class);
	 
	@Autowired
	private CategoryService categoryService;

	@Autowired
	private ObservableFeatureService observableFeatureService;
		
	@Autowired
	private OntologyTermService ontologyTermService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<CategoryResponse> createCategory(@Valid @RequestBody CategoryRequest categoryRequest)
			throws DatabaseException
	{
		return _createCategory(categoryRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CategoryResponse> createCategoryFromForm(@Valid @ModelAttribute CategoryRequest categoryRequest)
			throws DatabaseException
	{
		return _createCategory(categoryRequest);
	}

	private ResponseEntity<CategoryResponse> _createCategory(CategoryRequest categoryRequest) throws DatabaseException
	{
		Category category = categoryService.create(categoryRequest.toCategory());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/category/" + category.getId());
		return new ResponseEntity<CategoryResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public CategoryResponse retrieveCategory(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategory(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public CategoryResponse retrieveCategoryJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategory(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private CategoryResponse _retrieveCategory(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Category category = categoryService.read(id);
		if (category == null) throw new EntityNotFoundException("Category " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new CategoryResponse(category, expandFields);
	}
			
	@RequestMapping(value = "/{id}/observableFeature", method = RequestMethod.GET)
	public String retrieveCategoryXrefObservableFeature(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryXrefObservableFeature(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/observableFeature", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveCategoryXrefObservableFeatureJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryXrefObservableFeature(id, "json", expandFields);
	}
	
	private String _retrieveCategoryXrefObservableFeature(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Category category = categoryService.read(id);
		if (category == null) throw new EntityNotFoundException("Category " + id.toString() + " not found");
		Integer observableFeatureId = category.getObservableFeature_Id();
		String redirectUri = "redirect:/api/v1/observablefeature/" + observableFeatureId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	
	@RequestMapping(value = "/{id}/definition", method = RequestMethod.GET)
	public String retrieveCategoryXrefDefinition(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryXrefDefinition(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/definition", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveCategoryXrefDefinitionJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryXrefDefinition(id, "json", expandFields);
	}
	
	private String _retrieveCategoryXrefDefinition(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		Category category = categoryService.read(id);
		if (category == null) throw new EntityNotFoundException("Category " + id.toString() + " not found");
		Integer ontologyTermId = category.getDefinition_Id();
		String redirectUri = "redirect:/api/v1/ontologyterm/" + ontologyTermId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateCategory(@PathVariable Integer id, @Valid @RequestBody CategoryRequest categoryRequest)
			throws DatabaseException
	{
		_updateCategory(id, categoryRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CategoryResponse> updateCategoryFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute CategoryRequest categoryRequest) throws DatabaseException
	{
		return _createCategory(categoryRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCategoryPost(@PathVariable Integer id, @Valid @RequestBody CategoryRequest categoryRequest)
			throws DatabaseException
	{
		_updateCategory(id, categoryRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCategoryFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute CategoryRequest categoryRequest)
			throws DatabaseException
	{
		_updateCategory(id, categoryRequest);
	}

	private void _updateCategory(Integer id, CategoryRequest categoryRequest) throws DatabaseException
	{
		Category category = categoryRequest.toCategory();
		category.setId(id);
		categoryService.update(category);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCategory(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCategory(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCategoryPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCategory(id);
	}

	private void _deleteCategory(Integer id) throws DatabaseException
	{
		boolean isDeleted = categoryService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Category " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<CategoryResponse> retrieveCategoryCollection(@Valid EntityCollectionRequest categoryCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryCollection(categoryCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CategoryResponse> retrieveCategoryCollectionJson(@Valid EntityCollectionRequest categoryCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryCollection(categoryCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<CategoryResponse> retrieveCategoryCollectionPost(@Valid @RequestBody EntityCollectionRequest categoryCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryCollection(categoryCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CategoryResponse> retrieveCategoryCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest categoryCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCategoryCollection(categoryCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<CategoryResponse> _retrieveCategoryCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Category> categoryPager = categoryService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<CategoryResponse>(categoryPager, Lists.newArrayList(Iterables.transform(
				categoryPager.getIterable(), new Function<Category, CategoryResponse>()
				{
					@Override
					@Nullable
					public CategoryResponse apply(@Nullable Category category)
					{
						try
						{
							return category != null ? new CategoryResponse(category, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/category");
	}

	private static class CategoryRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer observableFeature;
		private String valueCode;
		private Integer definition;
		private Boolean isMissing = false;	
	
		public Category toCategory()
		{
			Category category = new Category();
			category.setIdentifier(identifier);
			category.setName(name);
			category.setDescription(description);
			category.setObservableFeature_Id(observableFeature);
			category.setValueCode(valueCode);
			category.setDefinition_Id(definition);
			category.setIsMissing(isMissing);
			return category;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setObservableFeature(Integer observableFeature)
		{
			this.observableFeature = observableFeature;
		}
		
		public void setValueCode(String valueCode)
		{
			this.valueCode = valueCode;
		}
		
		public void setDefinition(Integer definition)
		{
			this.definition = definition;
		}
		
		public void setIsMissing(Boolean isMissing)
		{
			this.isMissing = isMissing;
		}
		
	}

	static class CategoryResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object observableFeature;
		private final String valueCode;
		private final Object definition;
		private final Boolean isMissing;
	
		public CategoryResponse(Category category, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/category/" + category.getId();
			this.identifier = category.getIdentifier();
			this.name = category.getName();
			this.description = category.getDescription();
			if (expandFields != null && expandFields.contains("observableFeature")) this.observableFeature = new ObservableFeatureResponse(category.getObservableFeature(), null);
			else this.observableFeature = java.util.Collections.singletonMap("href", "/api/v1/category/" + category.getId() + "/observableFeature");
			this.valueCode = category.getValueCode();
			if (expandFields != null && expandFields.contains("definition")) this.definition = category.getDefinition() == null ? null : new OntologyTermResponse(category.getDefinition(), null);
			else this.definition = category.getDefinition() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/category/" + category.getId() + "/definition");
			this.isMissing = category.getIsMissing();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getObservableFeature()
		{
			return observableFeature;
		}
	
		public String getValueCode()
		{
			return valueCode;
		}
	
		public Object getDefinition()
		{
			return definition;
		}
	
		public Boolean getIsMissing()
		{
			return isMissing;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}