package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.Characteristic;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.CharacteristicService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/characteristic")
public class CharacteristicController
{
	private static Logger logger = Logger.getLogger(CharacteristicController.class);
	 
	@Autowired
	private CharacteristicService characteristicService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<CharacteristicResponse> createCharacteristic(@Valid @RequestBody CharacteristicRequest characteristicRequest)
			throws DatabaseException
	{
		return _createCharacteristic(characteristicRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CharacteristicResponse> createCharacteristicFromForm(@Valid @ModelAttribute CharacteristicRequest characteristicRequest)
			throws DatabaseException
	{
		return _createCharacteristic(characteristicRequest);
	}

	private ResponseEntity<CharacteristicResponse> _createCharacteristic(CharacteristicRequest characteristicRequest) throws DatabaseException
	{
		Characteristic characteristic = characteristicService.create(characteristicRequest.toCharacteristic());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/characteristic/" + characteristic.getId());
		return new ResponseEntity<CharacteristicResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public CharacteristicResponse retrieveCharacteristic(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCharacteristic(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public CharacteristicResponse retrieveCharacteristicJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCharacteristic(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private CharacteristicResponse _retrieveCharacteristic(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Characteristic characteristic = characteristicService.read(id);
		if (characteristic == null) throw new EntityNotFoundException("Characteristic " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new CharacteristicResponse(characteristic, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateCharacteristic(@PathVariable Integer id, @Valid @RequestBody CharacteristicRequest characteristicRequest)
			throws DatabaseException
	{
		_updateCharacteristic(id, characteristicRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<CharacteristicResponse> updateCharacteristicFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute CharacteristicRequest characteristicRequest) throws DatabaseException
	{
		return _createCharacteristic(characteristicRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCharacteristicPost(@PathVariable Integer id, @Valid @RequestBody CharacteristicRequest characteristicRequest)
			throws DatabaseException
	{
		_updateCharacteristic(id, characteristicRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateCharacteristicFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute CharacteristicRequest characteristicRequest)
			throws DatabaseException
	{
		_updateCharacteristic(id, characteristicRequest);
	}

	private void _updateCharacteristic(Integer id, CharacteristicRequest characteristicRequest) throws DatabaseException
	{
		Characteristic characteristic = characteristicRequest.toCharacteristic();
		characteristic.setId(id);
		characteristicService.update(characteristic);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCharacteristic(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCharacteristic(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteCharacteristicPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteCharacteristic(id);
	}

	private void _deleteCharacteristic(Integer id) throws DatabaseException
	{
		boolean isDeleted = characteristicService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Characteristic " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<CharacteristicResponse> retrieveCharacteristicCollection(@Valid EntityCollectionRequest characteristicCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCharacteristicCollection(characteristicCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CharacteristicResponse> retrieveCharacteristicCollectionJson(@Valid EntityCollectionRequest characteristicCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCharacteristicCollection(characteristicCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<CharacteristicResponse> retrieveCharacteristicCollectionPost(@Valid @RequestBody EntityCollectionRequest characteristicCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCharacteristicCollection(characteristicCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<CharacteristicResponse> retrieveCharacteristicCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest characteristicCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveCharacteristicCollection(characteristicCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<CharacteristicResponse> _retrieveCharacteristicCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Characteristic> characteristicPager = characteristicService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<CharacteristicResponse>(characteristicPager, Lists.newArrayList(Iterables.transform(
				characteristicPager.getIterable(), new Function<Characteristic, CharacteristicResponse>()
				{
					@Override
					@Nullable
					public CharacteristicResponse apply(@Nullable Characteristic characteristic)
					{
						try
						{
							return characteristic != null ? new CharacteristicResponse(characteristic, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/characteristic");
	}

	private static class CharacteristicRequest
	{
		private String identifier;
		private String name;
		private String description;
	
		public Characteristic toCharacteristic()
		{
			Characteristic characteristic = new Characteristic();
			characteristic.setIdentifier(identifier);
			characteristic.setName(name);
			characteristic.setDescription(description);
			return characteristic;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
	}

	static class CharacteristicResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
	
		public CharacteristicResponse(Characteristic characteristic, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/characteristic/" + characteristic.getId();
			this.identifier = characteristic.getIdentifier();
			this.name = characteristic.getName();
			this.description = characteristic.getDescription();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}