package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.EnvironmentalFactor;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.EnvironmentalFactorService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/environmentalfactor")
public class EnvironmentalFactorController
{
	private static Logger logger = Logger.getLogger(EnvironmentalFactorController.class);
	 
	@Autowired
	private EnvironmentalFactorService environmentalFactorService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<EnvironmentalFactorResponse> createEnvironmentalFactor(@Valid @RequestBody EnvironmentalFactorRequest environmentalFactorRequest)
			throws DatabaseException
	{
		return _createEnvironmentalFactor(environmentalFactorRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<EnvironmentalFactorResponse> createEnvironmentalFactorFromForm(@Valid @ModelAttribute EnvironmentalFactorRequest environmentalFactorRequest)
			throws DatabaseException
	{
		return _createEnvironmentalFactor(environmentalFactorRequest);
	}

	private ResponseEntity<EnvironmentalFactorResponse> _createEnvironmentalFactor(EnvironmentalFactorRequest environmentalFactorRequest) throws DatabaseException
	{
		EnvironmentalFactor environmentalFactor = environmentalFactorService.create(environmentalFactorRequest.toEnvironmentalFactor());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/environmentalfactor/" + environmentalFactor.getId());
		return new ResponseEntity<EnvironmentalFactorResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public EnvironmentalFactorResponse retrieveEnvironmentalFactor(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEnvironmentalFactor(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EnvironmentalFactorResponse retrieveEnvironmentalFactorJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEnvironmentalFactor(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private EnvironmentalFactorResponse _retrieveEnvironmentalFactor(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		EnvironmentalFactor environmentalFactor = environmentalFactorService.read(id);
		if (environmentalFactor == null) throw new EntityNotFoundException("EnvironmentalFactor " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EnvironmentalFactorResponse(environmentalFactor, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateEnvironmentalFactor(@PathVariable Integer id, @Valid @RequestBody EnvironmentalFactorRequest environmentalFactorRequest)
			throws DatabaseException
	{
		_updateEnvironmentalFactor(id, environmentalFactorRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<EnvironmentalFactorResponse> updateEnvironmentalFactorFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute EnvironmentalFactorRequest environmentalFactorRequest) throws DatabaseException
	{
		return _createEnvironmentalFactor(environmentalFactorRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateEnvironmentalFactorPost(@PathVariable Integer id, @Valid @RequestBody EnvironmentalFactorRequest environmentalFactorRequest)
			throws DatabaseException
	{
		_updateEnvironmentalFactor(id, environmentalFactorRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateEnvironmentalFactorFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute EnvironmentalFactorRequest environmentalFactorRequest)
			throws DatabaseException
	{
		_updateEnvironmentalFactor(id, environmentalFactorRequest);
	}

	private void _updateEnvironmentalFactor(Integer id, EnvironmentalFactorRequest environmentalFactorRequest) throws DatabaseException
	{
		EnvironmentalFactor environmentalFactor = environmentalFactorRequest.toEnvironmentalFactor();
		environmentalFactor.setId(id);
		environmentalFactorService.update(environmentalFactor);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteEnvironmentalFactor(@PathVariable Integer id) throws DatabaseException
	{
		_deleteEnvironmentalFactor(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteEnvironmentalFactorPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteEnvironmentalFactor(id);
	}

	private void _deleteEnvironmentalFactor(Integer id) throws DatabaseException
	{
		boolean isDeleted = environmentalFactorService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("EnvironmentalFactor " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<EnvironmentalFactorResponse> retrieveEnvironmentalFactorCollection(@Valid EntityCollectionRequest environmentalFactorCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEnvironmentalFactorCollection(environmentalFactorCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<EnvironmentalFactorResponse> retrieveEnvironmentalFactorCollectionJson(@Valid EntityCollectionRequest environmentalFactorCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEnvironmentalFactorCollection(environmentalFactorCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<EnvironmentalFactorResponse> retrieveEnvironmentalFactorCollectionPost(@Valid @RequestBody EntityCollectionRequest environmentalFactorCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEnvironmentalFactorCollection(environmentalFactorCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<EnvironmentalFactorResponse> retrieveEnvironmentalFactorCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest environmentalFactorCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveEnvironmentalFactorCollection(environmentalFactorCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<EnvironmentalFactorResponse> _retrieveEnvironmentalFactorCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<EnvironmentalFactor> environmentalFactorPager = environmentalFactorService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<EnvironmentalFactorResponse>(environmentalFactorPager, Lists.newArrayList(Iterables.transform(
				environmentalFactorPager.getIterable(), new Function<EnvironmentalFactor, EnvironmentalFactorResponse>()
				{
					@Override
					@Nullable
					public EnvironmentalFactorResponse apply(@Nullable EnvironmentalFactor environmentalFactor)
					{
						try
						{
							return environmentalFactor != null ? new EnvironmentalFactorResponse(environmentalFactor, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/environmentalfactor");
	}

	private static class EnvironmentalFactorRequest
	{
		private String identifier;
		private String name;
		private String description;
	
		public EnvironmentalFactor toEnvironmentalFactor()
		{
			EnvironmentalFactor environmentalFactor = new EnvironmentalFactor();
			environmentalFactor.setIdentifier(identifier);
			environmentalFactor.setName(name);
			environmentalFactor.setDescription(description);
			return environmentalFactor;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
	}

	static class EnvironmentalFactorResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
	
		public EnvironmentalFactorResponse(EnvironmentalFactor environmentalFactor, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/environmentalfactor/" + environmentalFactor.getId();
			this.identifier = environmentalFactor.getIdentifier();
			this.name = environmentalFactor.getName();
			this.description = environmentalFactor.getDescription();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}