package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.observ.value.IntValue;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.IntValueService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/intvalue")
public class IntValueController
{
	private static Logger logger = Logger.getLogger(IntValueController.class);
	 
	@Autowired
	private IntValueService intValueService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<IntValueResponse> createIntValue(@Valid @RequestBody IntValueRequest intValueRequest)
			throws DatabaseException
	{
		return _createIntValue(intValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<IntValueResponse> createIntValueFromForm(@Valid @ModelAttribute IntValueRequest intValueRequest)
			throws DatabaseException
	{
		return _createIntValue(intValueRequest);
	}

	private ResponseEntity<IntValueResponse> _createIntValue(IntValueRequest intValueRequest) throws DatabaseException
	{
		IntValue intValue = intValueService.create(intValueRequest.toIntValue());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/intvalue/" + intValue.getId());
		return new ResponseEntity<IntValueResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public IntValueResponse retrieveIntValue(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIntValue(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public IntValueResponse retrieveIntValueJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIntValue(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private IntValueResponse _retrieveIntValue(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		IntValue intValue = intValueService.read(id);
		if (intValue == null) throw new EntityNotFoundException("IntValue " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new IntValueResponse(intValue, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateIntValue(@PathVariable Integer id, @Valid @RequestBody IntValueRequest intValueRequest)
			throws DatabaseException
	{
		_updateIntValue(id, intValueRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<IntValueResponse> updateIntValueFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute IntValueRequest intValueRequest) throws DatabaseException
	{
		return _createIntValue(intValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateIntValuePost(@PathVariable Integer id, @Valid @RequestBody IntValueRequest intValueRequest)
			throws DatabaseException
	{
		_updateIntValue(id, intValueRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateIntValueFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute IntValueRequest intValueRequest)
			throws DatabaseException
	{
		_updateIntValue(id, intValueRequest);
	}

	private void _updateIntValue(Integer id, IntValueRequest intValueRequest) throws DatabaseException
	{
		IntValue intValue = intValueRequest.toIntValue();
		intValue.setId(id);
		intValueService.update(intValue);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteIntValue(@PathVariable Integer id) throws DatabaseException
	{
		_deleteIntValue(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteIntValuePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteIntValue(id);
	}

	private void _deleteIntValue(Integer id) throws DatabaseException
	{
		boolean isDeleted = intValueService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("IntValue " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<IntValueResponse> retrieveIntValueCollection(@Valid EntityCollectionRequest intValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIntValueCollection(intValueCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<IntValueResponse> retrieveIntValueCollectionJson(@Valid EntityCollectionRequest intValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIntValueCollection(intValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<IntValueResponse> retrieveIntValueCollectionPost(@Valid @RequestBody EntityCollectionRequest intValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIntValueCollection(intValueCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<IntValueResponse> retrieveIntValueCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest intValueCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveIntValueCollection(intValueCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<IntValueResponse> _retrieveIntValueCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<IntValue> intValuePager = intValueService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<IntValueResponse>(intValuePager, Lists.newArrayList(Iterables.transform(
				intValuePager.getIterable(), new Function<IntValue, IntValueResponse>()
				{
					@Override
					@Nullable
					public IntValueResponse apply(@Nullable IntValue intValue)
					{
						try
						{
							return intValue != null ? new IntValueResponse(intValue, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/intvalue");
	}

	private static class IntValueRequest
	{
		private Integer value;
	
		public IntValue toIntValue()
		{
			IntValue intValue = new IntValue();
			intValue.setValue(value);
			return intValue;
		}
		
		public void setValue(Integer value)
		{
			this.value = value;
		}
		
	}

	static class IntValueResponse
	{
		private final String href;
		private final Integer value;
	
		public IntValueResponse(IntValue intValue, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/intvalue/" + intValue.getId();
			this.value = intValue.getValue();
		}
	
		public String getHref()
		{
			return href;
		}
	
	
		public Integer getValue()
		{
			return value;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}