package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.Metabolite;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.MetaboliteService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/metabolite")
public class MetaboliteController
{
	private static Logger logger = Logger.getLogger(MetaboliteController.class);
	 
	@Autowired
	private MetaboliteService metaboliteService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<MetaboliteResponse> createMetabolite(@Valid @RequestBody MetaboliteRequest metaboliteRequest)
			throws DatabaseException
	{
		return _createMetabolite(metaboliteRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MetaboliteResponse> createMetaboliteFromForm(@Valid @ModelAttribute MetaboliteRequest metaboliteRequest)
			throws DatabaseException
	{
		return _createMetabolite(metaboliteRequest);
	}

	private ResponseEntity<MetaboliteResponse> _createMetabolite(MetaboliteRequest metaboliteRequest) throws DatabaseException
	{
		Metabolite metabolite = metaboliteService.create(metaboliteRequest.toMetabolite());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/metabolite/" + metabolite.getId());
		return new ResponseEntity<MetaboliteResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public MetaboliteResponse retrieveMetabolite(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMetabolite(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public MetaboliteResponse retrieveMetaboliteJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMetabolite(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private MetaboliteResponse _retrieveMetabolite(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		Metabolite metabolite = metaboliteService.read(id);
		if (metabolite == null) throw new EntityNotFoundException("Metabolite " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new MetaboliteResponse(metabolite, expandFields);
	}
			


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateMetabolite(@PathVariable Integer id, @Valid @RequestBody MetaboliteRequest metaboliteRequest)
			throws DatabaseException
	{
		_updateMetabolite(id, metaboliteRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<MetaboliteResponse> updateMetaboliteFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute MetaboliteRequest metaboliteRequest) throws DatabaseException
	{
		return _createMetabolite(metaboliteRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMetabolitePost(@PathVariable Integer id, @Valid @RequestBody MetaboliteRequest metaboliteRequest)
			throws DatabaseException
	{
		_updateMetabolite(id, metaboliteRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateMetaboliteFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute MetaboliteRequest metaboliteRequest)
			throws DatabaseException
	{
		_updateMetabolite(id, metaboliteRequest);
	}

	private void _updateMetabolite(Integer id, MetaboliteRequest metaboliteRequest) throws DatabaseException
	{
		Metabolite metabolite = metaboliteRequest.toMetabolite();
		metabolite.setId(id);
		metaboliteService.update(metabolite);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMetabolite(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMetabolite(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteMetabolitePost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteMetabolite(id);
	}

	private void _deleteMetabolite(Integer id) throws DatabaseException
	{
		boolean isDeleted = metaboliteService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("Metabolite " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<MetaboliteResponse> retrieveMetaboliteCollection(@Valid EntityCollectionRequest metaboliteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMetaboliteCollection(metaboliteCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MetaboliteResponse> retrieveMetaboliteCollectionJson(@Valid EntityCollectionRequest metaboliteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMetaboliteCollection(metaboliteCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<MetaboliteResponse> retrieveMetaboliteCollectionPost(@Valid @RequestBody EntityCollectionRequest metaboliteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMetaboliteCollection(metaboliteCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<MetaboliteResponse> retrieveMetaboliteCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest metaboliteCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveMetaboliteCollection(metaboliteCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<MetaboliteResponse> _retrieveMetaboliteCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<Metabolite> metabolitePager = metaboliteService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<MetaboliteResponse>(metabolitePager, Lists.newArrayList(Iterables.transform(
				metabolitePager.getIterable(), new Function<Metabolite, MetaboliteResponse>()
				{
					@Override
					@Nullable
					public MetaboliteResponse apply(@Nullable Metabolite metabolite)
					{
						try
						{
							return metabolite != null ? new MetaboliteResponse(metabolite, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/metabolite");
	}

	private static class MetaboliteRequest
	{
		private String identifier;
		private String name;
		private String description;
		private String formula;
		private Double mass;
		private String structure;
	
		public Metabolite toMetabolite()
		{
			Metabolite metabolite = new Metabolite();
			metabolite.setIdentifier(identifier);
			metabolite.setName(name);
			metabolite.setDescription(description);
			metabolite.setFormula(formula);
			metabolite.setMass(mass);
			metabolite.setStructure(structure);
			return metabolite;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setFormula(String formula)
		{
			this.formula = formula;
		}
		
		public void setMass(Double mass)
		{
			this.mass = mass;
		}
		
		public void setStructure(String structure)
		{
			this.structure = structure;
		}
		
	}

	static class MetaboliteResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final String formula;
		private final Double mass;
		private final String structure;
	
		public MetaboliteResponse(Metabolite metabolite, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/metabolite/" + metabolite.getId();
			this.identifier = metabolite.getIdentifier();
			this.name = metabolite.getName();
			this.description = metabolite.getDescription();
			this.formula = metabolite.getFormula();
			this.mass = metabolite.getMass();
			this.structure = metabolite.getStructure();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public String getFormula()
		{
			return formula;
		}
	
		public Double getMass()
		{
			return mass;
		}
	
		public String getStructure()
		{
			return structure;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}