package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.xgap.ProbeSet;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.ProbeSetService;
import org.molgenis.omx.xgap.Chromosome;
import org.molgenis.controller.ChromosomeController.ChromosomeResponse;
import org.molgenis.service.ChromosomeService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/probeset")
public class ProbeSetController
{
	private static Logger logger = Logger.getLogger(ProbeSetController.class);
	 
	@Autowired
	private ProbeSetService probeSetService;

	@Autowired
	private ChromosomeService chromosomeService;
		
	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<ProbeSetResponse> createProbeSet(@Valid @RequestBody ProbeSetRequest probeSetRequest)
			throws DatabaseException
	{
		return _createProbeSet(probeSetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProbeSetResponse> createProbeSetFromForm(@Valid @ModelAttribute ProbeSetRequest probeSetRequest)
			throws DatabaseException
	{
		return _createProbeSet(probeSetRequest);
	}

	private ResponseEntity<ProbeSetResponse> _createProbeSet(ProbeSetRequest probeSetRequest) throws DatabaseException
	{
		ProbeSet probeSet = probeSetService.create(probeSetRequest.toProbeSet());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/probeset/" + probeSet.getId());
		return new ResponseEntity<ProbeSetResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public ProbeSetResponse retrieveProbeSet(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSet(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public ProbeSetResponse retrieveProbeSetJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSet(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private ProbeSetResponse _retrieveProbeSet(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		ProbeSet probeSet = probeSetService.read(id);
		if (probeSet == null) throw new EntityNotFoundException("ProbeSet " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new ProbeSetResponse(probeSet, expandFields);
	}
			
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET)
	public String retrieveProbeSetXrefChromosome(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSetXrefChromosome(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/chromosome", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveProbeSetXrefChromosomeJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSetXrefChromosome(id, "json", expandFields);
	}
	
	private String _retrieveProbeSetXrefChromosome(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		ProbeSet probeSet = probeSetService.read(id);
		if (probeSet == null) throw new EntityNotFoundException("ProbeSet " + id.toString() + " not found");
		Integer chromosomeId = probeSet.getChromosome_Id();
		String redirectUri = "redirect:/api/v1/chromosome/" + chromosomeId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateProbeSet(@PathVariable Integer id, @Valid @RequestBody ProbeSetRequest probeSetRequest)
			throws DatabaseException
	{
		_updateProbeSet(id, probeSetRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<ProbeSetResponse> updateProbeSetFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute ProbeSetRequest probeSetRequest) throws DatabaseException
	{
		return _createProbeSet(probeSetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProbeSetPost(@PathVariable Integer id, @Valid @RequestBody ProbeSetRequest probeSetRequest)
			throws DatabaseException
	{
		_updateProbeSet(id, probeSetRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateProbeSetFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute ProbeSetRequest probeSetRequest)
			throws DatabaseException
	{
		_updateProbeSet(id, probeSetRequest);
	}

	private void _updateProbeSet(Integer id, ProbeSetRequest probeSetRequest) throws DatabaseException
	{
		ProbeSet probeSet = probeSetRequest.toProbeSet();
		probeSet.setId(id);
		probeSetService.update(probeSet);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProbeSet(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProbeSet(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteProbeSetPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteProbeSet(id);
	}

	private void _deleteProbeSet(Integer id) throws DatabaseException
	{
		boolean isDeleted = probeSetService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("ProbeSet " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<ProbeSetResponse> retrieveProbeSetCollection(@Valid EntityCollectionRequest probeSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSetCollection(probeSetCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProbeSetResponse> retrieveProbeSetCollectionJson(@Valid EntityCollectionRequest probeSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSetCollection(probeSetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<ProbeSetResponse> retrieveProbeSetCollectionPost(@Valid @RequestBody EntityCollectionRequest probeSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSetCollection(probeSetCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<ProbeSetResponse> retrieveProbeSetCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest probeSetCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveProbeSetCollection(probeSetCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<ProbeSetResponse> _retrieveProbeSetCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<ProbeSet> probeSetPager = probeSetService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<ProbeSetResponse>(probeSetPager, Lists.newArrayList(Iterables.transform(
				probeSetPager.getIterable(), new Function<ProbeSet, ProbeSetResponse>()
				{
					@Override
					@Nullable
					public ProbeSetResponse apply(@Nullable ProbeSet probeSet)
					{
						try
						{
							return probeSet != null ? new ProbeSetResponse(probeSet, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/probeset");
	}

	private static class ProbeSetRequest
	{
		private String identifier;
		private String name;
		private String description;
		private Integer chromosome;
		private Double cM;
		private Long bpStart;
		private Long bpEnd;
		private String seq;
		private String symbol;
	
		public ProbeSet toProbeSet()
		{
			ProbeSet probeSet = new ProbeSet();
			probeSet.setIdentifier(identifier);
			probeSet.setName(name);
			probeSet.setDescription(description);
			probeSet.setChromosome_Id(chromosome);
			probeSet.setCM(cM);
			probeSet.setBpStart(bpStart);
			probeSet.setBpEnd(bpEnd);
			probeSet.setSeq(seq);
			probeSet.setSymbol(symbol);
			return probeSet;
		}
		
		public void setIdentifier(String identifier)
		{
			this.identifier = identifier;
		}
		
		public void setName(String name)
		{
			this.name = name;
		}
		
		public void setDescription(String description)
		{
			this.description = description;
		}
		
		public void setChromosome(Integer chromosome)
		{
			this.chromosome = chromosome;
		}
		
		public void setCM(Double cM)
		{
			this.cM = cM;
		}
		
		public void setBpStart(Long bpStart)
		{
			this.bpStart = bpStart;
		}
		
		public void setBpEnd(Long bpEnd)
		{
			this.bpEnd = bpEnd;
		}
		
		public void setSeq(String seq)
		{
			this.seq = seq;
		}
		
		public void setSymbol(String symbol)
		{
			this.symbol = symbol;
		}
		
	}

	static class ProbeSetResponse
	{
		private final String href;
		private final String identifier;
		private final String name;
		private final String description;
		private final Object chromosome;
		private final Double cM;
		private final Long bpStart;
		private final Long bpEnd;
		private final String seq;
		private final String symbol;
	
		public ProbeSetResponse(ProbeSet probeSet, final Set<String> expandFields) throws DatabaseException
		{
			this.href = "/api/v1/probeset/" + probeSet.getId();
			this.identifier = probeSet.getIdentifier();
			this.name = probeSet.getName();
			this.description = probeSet.getDescription();
			if (expandFields != null && expandFields.contains("chromosome")) this.chromosome = probeSet.getChromosome() == null ? null : new ChromosomeResponse(probeSet.getChromosome(), null);
			else this.chromosome = probeSet.getChromosome() == null ? null : java.util.Collections.singletonMap("href", "/api/v1/probeset/" + probeSet.getId() + "/chromosome");
			this.cM = probeSet.getCM();
			this.bpStart = probeSet.getBpStart();
			this.bpEnd = probeSet.getBpEnd();
			this.seq = probeSet.getSeq();
			this.symbol = probeSet.getSymbol();
		}
	
		public String getHref()
		{
			return href;
		}
	
		public String getIdentifier()
		{
			return identifier;
		}
	
		public String getName()
		{
			return name;
		}
	
	
		public String getDescription()
		{
			return description;
		}
	
		public Object getChromosome()
		{
			return chromosome;
		}
	
		public Double getCM()
		{
			return cM;
		}
	
		public Long getBpStart()
		{
			return bpStart;
		}
	
		public Long getBpEnd()
		{
			return bpEnd;
		}
	
		public String getSeq()
		{
			return seq;
		}
	
		public String getSymbol()
		{
			return symbol;
		}
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}