package org.molgenis.controller;

import java.beans.PropertyEditorSupport;
import java.text.ParseException;
import java.lang.RuntimeException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.Valid;

import org.apache.log4j.Logger;

import org.molgenis.omx.auth.UserAuthority;
import org.molgenis.framework.server.EntityCollectionRequest;
import org.molgenis.framework.server.EntityCollectionResponse;
import org.molgenis.framework.db.DatabaseAccessException;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.EntityNotFoundException;
import org.molgenis.framework.db.QueryRule;
import org.molgenis.service.UserAuthorityService;
import org.molgenis.util.EntityPager;
import org.molgenis.util.ErrorMessageResponse;
import org.molgenis.util.ErrorMessageResponse.ErrorMessage;
import org.molgenis.util.MolgenisDateFormat;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import org.apache.commons.lang3.StringUtils;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@SuppressWarnings("unused")
@Lazy
@Controller
@RequestMapping("/api/v1/userauthority")
public class UserAuthorityController
{
	private static Logger logger = Logger.getLogger(UserAuthorityController.class);
	 
	@Autowired
	private UserAuthorityService userAuthorityService;

	@RequestMapping(method = RequestMethod.POST)
	@ResponseBody
	public ResponseEntity<UserAuthorityResponse> createUserAuthority(@Valid @RequestBody UserAuthorityRequest userAuthorityRequest)
			throws DatabaseException
	{
		return _createUserAuthority(userAuthorityRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(method = RequestMethod.POST, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<UserAuthorityResponse> createUserAuthorityFromForm(@Valid @ModelAttribute UserAuthorityRequest userAuthorityRequest)
			throws DatabaseException
	{
		return _createUserAuthority(userAuthorityRequest);
	}

	private ResponseEntity<UserAuthorityResponse> _createUserAuthority(UserAuthorityRequest userAuthorityRequest) throws DatabaseException
	{
		UserAuthority userAuthority = userAuthorityService.create(userAuthorityRequest.toUserAuthority());
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Location", "/api/v1/userauthority/" + userAuthority.getId());
		return new ResponseEntity<UserAuthorityResponse>(responseHeaders, HttpStatus.CREATED);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.GET)
	@ResponseBody
	public UserAuthorityResponse retrieveUserAuthority(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthority(id, expandFields);
	}
		
	@RequestMapping(value = "/{id}", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public UserAuthorityResponse retrieveUserAuthorityJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthority(id, expandFields);
	}

	@InitBinder
	public void binder(WebDataBinder binder)
	{

		binder.registerCustomEditor(Date.class, new PropertyEditorSupport()
		{
			@Override
			public void setAsText(String value)
			{
				try
				{
					if (StringUtils.isNotBlank(value))
					{
						setValue(MolgenisDateFormat.getDateFormat().parse(value));
					}
				}
				catch (ParseException e)
				{
					throw new RuntimeException(e);
				}
			}

			@Override
			public String getAsText()
			{
				if (getValue() == null)
				{
					return null;
				}
				
				return MolgenisDateFormat.getDateFormat().format((Date) getValue());
			}

		});
	}

	private UserAuthorityResponse _retrieveUserAuthority(Integer id, String... expandFieldsStr) throws DatabaseException
	{
		UserAuthority userAuthority = userAuthorityService.read(id);
		if (userAuthority == null) throw new EntityNotFoundException("UserAuthority " + id.toString() + " not found");
		Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new UserAuthorityResponse(userAuthority, expandFields);
	}
			
	@RequestMapping(value = "/{id}/molgenisUser", method = RequestMethod.GET)
	public String retrieveUserAuthorityXrefMolgenisUser(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthorityXrefMolgenisUser(id, null, expandFields);
	}
	
	@RequestMapping(value = "/{id}/molgenisUser", method = RequestMethod.GET, params = "format=json", produces = "application/json")
	public String retrieveUserAuthorityXrefMolgenisUserJson(@PathVariable Integer id, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthorityXrefMolgenisUser(id, "json", expandFields);
	}
	
	private String _retrieveUserAuthorityXrefMolgenisUser(Integer id, String format, String... expandFieldsStr) throws DatabaseException
	{
		UserAuthority userAuthority = userAuthorityService.read(id);
		if (userAuthority == null) throw new EntityNotFoundException("UserAuthority " + id.toString() + " not found");
		Integer molgenisUserId = userAuthority.getMolgenisUser_Id();
		String redirectUri = "redirect:/api/v1/molgenisuser/" + molgenisUserId.toString();
		StringBuilder qsBuilder = new StringBuilder();
		if(format != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("format=").append(format);
		if(expandFieldsStr != null) qsBuilder.append(qsBuilder.length() == 0 ? '?' : '&').append("expand=").append(Joiner.on(',').join(expandFieldsStr));
		return qsBuilder.length() == 0 ? redirectUri : redirectUri + qsBuilder.toString();
	}
	


	@RequestMapping(value = "/{id}", method = RequestMethod.PUT)
	@ResponseStatus(HttpStatus.OK)
	public void updateUserAuthority(@PathVariable Integer id, @Valid @RequestBody UserAuthorityRequest userAuthorityRequest)
			throws DatabaseException
	{
		_updateUserAuthority(id, userAuthorityRequest);
	}

	// Spring's FormHttpMessageConverter cannot bind target classes (as ModelAttribute can)
	@RequestMapping(value = "/{id}", method = RequestMethod.PUT, headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseBody
	public ResponseEntity<UserAuthorityResponse> updateUserAuthorityFromForm(@PathVariable Integer id, @PathVariable String _method,
			@Valid @ModelAttribute UserAuthorityRequest userAuthorityRequest) throws DatabaseException
	{
		return _createUserAuthority(userAuthorityRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateUserAuthorityPost(@PathVariable Integer id, @Valid @RequestBody UserAuthorityRequest userAuthorityRequest)
			throws DatabaseException
	{
		_updateUserAuthority(id, userAuthorityRequest);
	}

	// Tunnel PUT through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=PUT", headers = "Content-Type=application/x-www-form-urlencoded")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void updateUserAuthorityFromFormPost(@PathVariable Integer id, @Valid @ModelAttribute UserAuthorityRequest userAuthorityRequest)
			throws DatabaseException
	{
		_updateUserAuthority(id, userAuthorityRequest);
	}

	private void _updateUserAuthority(Integer id, UserAuthorityRequest userAuthorityRequest) throws DatabaseException
	{
		UserAuthority userAuthority = userAuthorityRequest.toUserAuthority();
		userAuthority.setId(id);
		userAuthorityService.update(userAuthority);
	}

	@RequestMapping(value = "/{id}", method = RequestMethod.DELETE)
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteUserAuthority(@PathVariable Integer id) throws DatabaseException
	{
		_deleteUserAuthority(id);
	}

	// Tunnel DELETE through POST
	@RequestMapping(value = "/{id}", method = RequestMethod.POST, params = "_method=DELETE")
	@ResponseStatus(HttpStatus.NO_CONTENT)
	public void deleteUserAuthorityPost(@PathVariable Integer id) throws DatabaseException
	{
		_deleteUserAuthority(id);
	}

	private void _deleteUserAuthority(Integer id) throws DatabaseException
	{
		boolean isDeleted = userAuthorityService.deleteById(id);
		if(!isDeleted) throw new EntityNotFoundException("UserAuthority " + id.toString() + " not found");
	}
	
	@RequestMapping(method = RequestMethod.GET)
	@ResponseBody
	public EntityCollectionResponse<UserAuthorityResponse> retrieveUserAuthorityCollection(@Valid EntityCollectionRequest userAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthorityCollection(userAuthorityCollectionRequest, expandFields);
	}

	@RequestMapping(method = RequestMethod.GET, params = "format=json", produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<UserAuthorityResponse> retrieveUserAuthorityCollectionJson(@Valid EntityCollectionRequest userAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthorityCollection(userAuthorityCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = "_method=GET")
	@ResponseBody
	public EntityCollectionResponse<UserAuthorityResponse> retrieveUserAuthorityCollectionPost(@Valid @RequestBody EntityCollectionRequest userAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthorityCollection(userAuthorityCollectionRequest, expandFields);
	}

	// Tunnel GET with body through POST
	@RequestMapping(method = RequestMethod.POST, params = {"_method=GET", "format=json"}, produces = "application/json")
	@ResponseBody
	public EntityCollectionResponse<UserAuthorityResponse> retrieveUserAuthorityCollectionJsonPost(@Valid @RequestBody EntityCollectionRequest userAuthorityCollectionRequest, @RequestParam(value="expand", required=false) String... expandFields) throws DatabaseException
	{
		return _retrieveUserAuthorityCollection(userAuthorityCollectionRequest, expandFields);
	}
	
	private EntityCollectionResponse<UserAuthorityResponse> _retrieveUserAuthorityCollection(EntityCollectionRequest entityCollectionRequest, String... expandFieldsStr) throws DatabaseException
	{
		EntityPager<UserAuthority> userAuthorityPager = userAuthorityService.readAll(entityCollectionRequest.getStart(), entityCollectionRequest.getNum(), entityCollectionRequest.getQ());
		final Set<String> expandFields = expandFieldsStr != null ? new HashSet<String>(Arrays.asList(expandFieldsStr)) : null;
		return new EntityCollectionResponse<UserAuthorityResponse>(userAuthorityPager, Lists.newArrayList(Iterables.transform(
				userAuthorityPager.getIterable(), new Function<UserAuthority, UserAuthorityResponse>()
				{
					@Override
					@Nullable
					public UserAuthorityResponse apply(@Nullable UserAuthority userAuthority)
					{
						try
						{
							return userAuthority != null ? new UserAuthorityResponse(userAuthority, expandFields) : null;
						} catch(DatabaseException e)
						{
							throw new RuntimeException(e);
						}
					}
				})), "/api/v1/userauthority");
	}

	private static class UserAuthorityRequest
	{
		private String role;
	
		public UserAuthority toUserAuthority()
		{
			UserAuthority userAuthority = new UserAuthority();
			userAuthority.setRole(role);
			return userAuthority;
		}
		
		public void setRole(String role)
		{
			this.role = role;
		}
		
		
	}

	static class UserAuthorityResponse
	{
		private final String role;
		private final String href;
	
		public UserAuthorityResponse(UserAuthority userAuthority, final Set<String> expandFields) throws DatabaseException
		{
			this.role = userAuthority.getRole();
			this.href = "/api/v1/userauthority/" + userAuthority.getId();
		}
	
		public String getRole()
		{
			return role;
		}
	
		public String getHref()
		{
			return href;
		}
	
	
	}
	
	@ExceptionHandler(EntityNotFoundException.class)
	@ResponseStatus(value = HttpStatus.NOT_FOUND)
	@ResponseBody
	public ErrorMessageResponse handleEntityNotFoundException(EntityNotFoundException e)
	{
		logger.debug(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseException(DatabaseException e)
	{
		logger.error(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}

	@ExceptionHandler(DatabaseAccessException.class)
	@ResponseStatus(value = HttpStatus.UNAUTHORIZED)
	@ResponseBody
	public ErrorMessageResponse handleDatabaseAccessException(DatabaseAccessException e)
	{
		logger.info(e);
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
	
	@ExceptionHandler(RuntimeException.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	@ResponseBody
	public ErrorMessageResponse handleRuntimeException(RuntimeException e)
	{
		logger.error(e);		
		return new ErrorMessageResponse(new ErrorMessage(e.getMessage()));
	}
}