package org.molgenis.util.tuple;

import java.sql.Date;
import java.sql.Timestamp;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.log4j.Logger;
import org.molgenis.util.AbstractEntity;
import org.molgenis.util.ListEscapeUtils;

/**
 * Tuple that delegates all calls to Tuple.get
 */
@edu.umd.cs.findbugs.annotations.SuppressWarnings(value = "NP_BOOLEAN_RETURN_NULL", justification = "return true/false/null on purpose")
public abstract class AbstractTuple implements Tuple
{
	private static final long serialVersionUID = 1L;
	private static final Logger logger = Logger.getLogger(AbstractTuple.class);

	@Override
	public boolean hasColNames()
	{
		Iterable<String> it = getColNames();
		return it != null ? it.iterator().hasNext() : false;
	}

	@Override
	public boolean isNull(String colName)
	{
		return get(colName) == null;
	}

	@Override
	public boolean isNull(int col)
	{
		return get(col) == null;
	}

	@Override
	public String getString(String colName)
	{
		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof String) return (String) obj;
		else return obj.toString();
	}

	@Override
	public String getString(int col)
	{
		Object obj = get(col);
		if (obj == null) return null;
		else if (obj instanceof String) return (String) obj;
		else return obj.toString();
	}

	@Override
	public Integer getInt(String colName)
	{
		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof Integer) return (Integer) obj;
		else return Integer.parseInt(obj.toString());
	}

	@Override
	public Integer getInt(int col)
	{
		Object obj = get(col);
		if (obj == null) return null;
		else if (obj instanceof Integer) return (Integer) obj;
		else return Integer.parseInt(obj.toString());
	}

	@Override
	public Long getLong(String colName)
	{
		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof Long) return (Long) obj;
		else return Long.parseLong(obj.toString());
	}

	@Override
	public Long getLong(int col)
	{
		Object obj = get(col);
		if (obj == null) return null;
		else if (obj instanceof Long) return (Long) obj;
		else return Long.parseLong(obj.toString());
	}

	@Override
	public Boolean getBoolean(String colName)
	{
		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof Boolean) return (Boolean) obj;
		else
		{
			String str = obj.toString();
			return str.equalsIgnoreCase("true") || str.equalsIgnoreCase("1");
		}
	}

	@Override
	public Boolean getBoolean(int col)
	{
		Object obj = get(col);
		if (obj == null) return null;
		else if (obj instanceof Boolean) return (Boolean) obj;
		else
		{
			String str = obj.toString();
			return str.equalsIgnoreCase("true") || str.equalsIgnoreCase("1");
		}
	}

	@Override
	public Double getDouble(String colName)
	{
		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof Double) return (Double) obj; // FIXME
		else return Double.parseDouble(obj.toString());
	}

	@Override
	public Double getDouble(int col)
	{
		Object obj = get(col);
		if (obj == null) return null;
		else if (obj instanceof Double) return (Double) obj;
		else return Double.parseDouble(obj.toString());
	}

	@Override
	public Date getDate(String colName)
	{
		Object obj = get(colName);
		return objToDate(obj);
	}

	@Override
	public Date getDate(int col)
	{
		Object obj = get(col);
		return objToDate(obj);
	}

	private Date objToDate(Object obj)
	{
		Date result = null;
		if (obj == null) return null;
		else if (obj instanceof Date) return (Date) obj;
		else try
		{
			try
			{
				result = Date.valueOf(obj.toString());
			}
			catch (IllegalArgumentException e)
			{
				result = AbstractEntity.string2date(obj.toString());
			}
		}
		catch (ParseException e)
		{
			logger.warn("unable to parse input tot SQL Date: " + obj.toString());
		}
		return result;
	}

	@Override
	public Timestamp getTimestamp(String colName)
	{
		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof Timestamp) return (Timestamp) obj;
		else if (obj instanceof Date) return new Timestamp(((Date) obj).getTime());
		else return Timestamp.valueOf(obj.toString());
	}

	@Override
	public Timestamp getTimestamp(int col)
	{
		Object obj = get(col);
		if (obj == null) return null;
		else if (obj instanceof Timestamp) return (Timestamp) obj;
		else if (obj instanceof Date) return new Timestamp(((Date) obj).getTime());
		else return Timestamp.valueOf(obj.toString());
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<String> getList(String colName)
	{

		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof List<?>) return (List<String>) obj;
		else if (obj instanceof String) return ListEscapeUtils.toList((String) obj);
		else return ListEscapeUtils.toList(obj.toString());
	}

	// TODO: Improve code of getIntList (maybe merge with getList above?)
	@SuppressWarnings("unchecked")
	@Override
	public List<Integer> getIntList(String colName)
	{

		Object obj = get(colName);
		if (obj == null) return null;
		else if (obj instanceof List<?>)
		{
			// it seems we need explicit cast to Int sometimes
			ArrayList<Integer> intList = new ArrayList<Integer>();
			for (Object o : (List<?>) obj)
			{
				if (o instanceof Integer) intList.add((Integer) o);
				else if (o instanceof String) intList.add(Integer.parseInt((String) o));
			}
			return intList;
		}
		else if (obj instanceof Integer) return new ArrayList<Integer>(Arrays.asList((Integer) obj));
		else return null;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<String> getList(int col)
	{
		Object obj = get(col);
		if (obj == null) return null;
		else if (obj instanceof List<?>) return (List<String>) obj;
		else if (obj instanceof String) return ListEscapeUtils.toList((String) obj);
		else return ListEscapeUtils.toList(obj.toString());
	}

	@Override
	public boolean isEmpty()
	{
		for (String colName : getColNames())
		{
			if (!isNull(colName))
			{
				return false;
			}
		}

		return true;

	}

	@Override
	public String toString()
	{
		StringBuilder strBuilder = new StringBuilder();
		for (String colName : getColNames())
			strBuilder.append(colName).append('=').append(get(colName)).append(',');
		if (strBuilder.length() > 0) strBuilder.deleteCharAt(strBuilder.length() - 1);

		return "Tuple[" + strBuilder.toString() + ']';
	}
}
