
/* File:        org.molgenis.omx/model/Chromosome.java
 * Copyright:   GBIC 2000-2013, all rights reserved
 * Date:        November 22, 2013
 * 
 * generator:   org.molgenis.generators.db.EntityImporterGen 4.0.0-testing
 *
 * 
 * THIS FILE HAS BEEN GENERATED, PLEASE DO NOT EDIT!
 */

package org.molgenis.omx.xgap.db;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang3.StringUtils;
import com.google.common.collect.Sets;
import org.apache.log4j.Logger;
import org.molgenis.data.Entity;
import org.molgenis.data.Repository;
import org.molgenis.framework.db.Database;
import org.molgenis.framework.db.DatabaseException;
import org.molgenis.framework.db.Database.DatabaseAction;
import org.molgenis.framework.db.EntityImporter;
import org.molgenis.io.processor.LowerCaseProcessor;
import org.molgenis.io.TupleReader;
import org.molgenis.util.tuple.Tuple;

import org.molgenis.omx.observ.target.OntologyTerm;
import org.molgenis.omx.xgap.Chromosome;


/**
 * Reads Chromosome from a delimited (csv) file, resolving xrefs to ids where needed, that is the tricky bit ;-)
 */
public class ChromosomeEntityImporter implements EntityImporter
{
	private static final Logger logger = Logger.getLogger(ChromosomeEntityImporter.class);
	
	private static int BATCH_SIZE = 10000;
	//foreign key map for xref 'species' (maps ontologyTerm.Identifier -> ontologyTerm.id)			
	final Map<String,Integer> speciesKeymap = new TreeMap<String,Integer>();	
			
	/**
	 * Imports Chromosome from tab/comma delimited File
	 * @param db database to import into
	 * @param reader csv reader to load data from
	 * @param defaults to set default values for each row
	 * @param dbAction indicating wether to add,update,remove etc
	 * @param missingValues indicating what value in the csv is treated as 'null' (e.g. "" or "NA")
	 * @return number of elements imported
	 */
	@Override
	public int importEntity(Repository<? extends Entity> repository, Database db, DatabaseAction dbAction) throws IOException, DatabaseException 
	{
		//wrapper to count
		final AtomicInteger total = new AtomicInteger(0);
	try {
		//cache for entities of which xrefs couldn't be resolved (e.g. if there is a self-refence)
		//these entities can be updated with their xrefs in a second round when all entities are in the database
		List<Chromosome> chromosomesMissingRefs = new ArrayList<Chromosome>();
	
		//cache for objects to be imported from file (in batch)
		final List<Chromosome> chromosomeList = new ArrayList<Chromosome>(BATCH_SIZE); // FIXME
		
		for(Entity entity : repository)
		{
			// skip empty rows
			if (!hasValues(entity)) continue;
			
			//parse object, setting defaults and values from file
			Chromosome object = new Chromosome();
			object.set(entity, false);				
			chromosomeList.add(object);		
			
			//add to db when batch size is reached
			if(chromosomeList.size() == BATCH_SIZE)
			{
				//resolve foreign keys and copy those entities that could not be resolved to the missingRefs list
				chromosomesMissingRefs.addAll(resolveForeignKeys(db, chromosomeList));
				chromosomeList.removeAll(chromosomesMissingRefs);
				
				//update objects in the database using xref_label defined secondary key(s) 'Identifier' defined in xref_label
				db.update(chromosomeList,dbAction, "Identifier");
				
				//clear for next batch						
				chromosomeList.clear();		
				
				//keep count
				total.set(total.get() + BATCH_SIZE);
				
				db.getEntityManager().flush();
				db.getEntityManager().clear();		
			}
		}
			
		//add remaining elements to the database
		if(!chromosomeList.isEmpty())
		{
			total.set(total.get() + chromosomeList.size());
			
			//resolve foreign keys, again keeping track of those entities that could not be solved
			chromosomesMissingRefs.addAll(resolveForeignKeys(db, chromosomeList));
			chromosomeList.removeAll(chromosomesMissingRefs);
			
			//update objects in the database using xref_label defined secondary key(s) 'Identifier' defined in xref_label
			db.update(chromosomeList,dbAction, "Identifier");
		}
		
		//Try to resolve FK's for entities until all are resolved or we have more then 100 iterations
		List<Chromosome> chromosomes = new ArrayList<Chromosome>(chromosomesMissingRefs);

		int iterationCount = 0;

		do
		{
			chromosomesMissingRefs = resolveForeignKeys(db, chromosomesMissingRefs);
			
			LinkedHashSet<Chromosome> differenceSet = new LinkedHashSet<Chromosome>();
			Sets.symmetricDifference(new LinkedHashSet<Chromosome>(chromosomes), new LinkedHashSet<Chromosome>(chromosomesMissingRefs)).copyInto(differenceSet);
			List<Chromosome> resolvablechromosomes = new ArrayList<Chromosome>(differenceSet);
			
			chromosomes.removeAll(resolvablechromosomes);
			
			db.update(resolvablechromosomes,dbAction, "Identifier");

			if (iterationCount++ > 100)
			{
			String identifier = "";
			String name = "";
				for(Chromosome blaat : chromosomesMissingRefs){
					identifier = blaat.getValues().get("Identifier").toString();
					name = blaat.getValues().get("Name").toString();
				}
				throw new Exception(
						"Import of 'chromosome' entity failed:"
								+ "This is probably caused by a(n) 'chromosome' that has a reference but that does not exist."
								+"(identifier:"+identifier+", name:"+name+")");		
			}
		}
		while (chromosomesMissingRefs.size() > 0);

		logger.info("imported " + total.get() + " chromosome from CSV");

		} catch(Exception e) {throw new IOException(e);}
		return total.get();
	}	
	
	private boolean hasValues(Entity entity)
	{
		for (String attributeName : entity.getAttributeNames())
		{
			if (entity.get(attributeName) != null) return true;
		}
		return false;
	}
	
	/**
	 * This method tries to resolve foreign keys (i.e. xref_field) based on the secondary key/key (i.e. xref_labels).
	 *
	 * @param db database
	 * @param chromosomeList 
	 * @return the entities for which foreign keys cannot be resolved
	 */
	private List<Chromosome> resolveForeignKeys(Database db, List<Chromosome> chromosomeList) throws Exception
	{
		//keep a list of Chromosome instances that miss a reference which might be resolvable later
		List<Chromosome> chromosomesMissingRefs = new ArrayList<Chromosome>();
	
		//resolve xref 'species' from ontologyTerm.Identifier -> ontologyTerm.id
		for(Chromosome o: chromosomeList) 
		{
			if(o.getSpecies_Identifier() != null) 
				speciesKeymap.put(o.getSpecies_Identifier(), null);
		}
		
		if(speciesKeymap.size() > 0) 
		{
			List<OntologyTerm> speciesList = db.query(OntologyTerm.class).in("Identifier",new ArrayList<Object>(speciesKeymap.keySet())).find();
			for(OntologyTerm xref :  speciesList)
			{
				speciesKeymap.put(xref.getIdentifier(), xref.getId());
			}
		}
		//update objects with foreign key values
		for(Chromosome o:  chromosomeList)
		{
			while(true){
				//update xref Species
				if(o.getSpecies_Identifier() != null) 
				{
					String key = o.getSpecies_Identifier();
					if(speciesKeymap.get(key) == null)
					{
						throw new Exception("Import of 'Chromosome' objects failed: cannot find OntologyTerm for species_Identifier='"+o.getSpecies_Identifier()+"'");
					}
					o.setSpecies_Id(speciesKeymap.get(key));
				}
				break;
			}
		}
		
		speciesKeymap.clear();
		
		return chromosomesMissingRefs;
	}
}

