package org.molgenis.omx.catalogmanager;

import org.molgenis.catalog.Catalog;
import org.molgenis.catalog.CatalogMeta;
import org.molgenis.catalog.UnknownCatalogException;
import org.molgenis.catalogmanager.CatalogManagerService;
import org.molgenis.data.DataService;
import org.molgenis.data.support.QueryImpl;
import org.molgenis.omx.observ.Protocol;
import org.molgenis.omx.study.StudyDataRequest;
import org.molgenis.study.UnknownStudyDefinitionException;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import java.util.Collections;
import java.util.UUID;

public class OmxCatalogManagerService implements CatalogManagerService
{
	private final DataService dataService;

	public OmxCatalogManagerService(DataService dataService)
	{
		if (dataService == null) throw new IllegalArgumentException("dataService is null");
		this.dataService = dataService;
	}

	@Override
	public Iterable<CatalogMeta> getCatalogs()
	{
		Iterable<Protocol> protocols = dataService.findAll(Protocol.ENTITY_NAME,
				new QueryImpl().eq(Protocol.ROOT, true), Protocol.class);

		return Iterables.transform(protocols, new Function<Protocol, CatalogMeta>()
		{
			@Override
			public CatalogMeta apply(Protocol protocol)
			{
				CatalogMeta catalogMeta = new CatalogMeta(protocol.getId().toString(), protocol.getName());
				catalogMeta.setDescription(protocol.getDescription());
				return catalogMeta;
			}
		});
	}

	@Override
	public Catalog getCatalog(String id) throws UnknownCatalogException
	{
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
        //workaround for getting the rendering correct in the catalog manager, without breaking the studymanager
        Protocol root = new Protocol();
        root.setName(protocol.getName());
        root.setId(-1);
        root.setIdentifier(UUID.randomUUID().toString());
        root.setSubprotocols(Collections.singletonList(protocol));
		return new OmxCatalog(root, dataService);
	}

	@Override
	public Catalog getCatalogOfStudyDefinition(String id) throws UnknownCatalogException,
			UnknownStudyDefinitionException
	{
		StudyDataRequest studyDataRequest = dataService.findOne(StudyDataRequest.ENTITY_NAME,
				new QueryImpl().eq(StudyDataRequest.ID, Integer.valueOf(id)), StudyDataRequest.class);
		if (studyDataRequest == null) throw new UnknownStudyDefinitionException("Study definition [" + id
				+ "] does not exist");
		Protocol protocol = studyDataRequest.getProtocol();
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
		return new OmxCatalog(protocol, dataService);
	}

	@Override
	public void loadCatalog(String id) throws UnknownCatalogException
	{
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
		setProtocolActive(protocol, true);
	}

	@Override
	public void unloadCatalog(String id) throws UnknownCatalogException
	{
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
		setProtocolActive(protocol, false);
	}

	@Override
	public boolean isCatalogLoaded(String id) throws UnknownCatalogException
	{
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
		return protocol.getActive();
	}

	@Override
	public boolean isCatalogActivated(String id) throws UnknownCatalogException
	{
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
		return protocol.getActive();
	}

	@Override
	public void loadCatalogOfStudyDefinition(String id) throws UnknownCatalogException, UnknownStudyDefinitionException
	{
		StudyDataRequest studyDataRequest = dataService.findOne(StudyDataRequest.ENTITY_NAME,
				new QueryImpl().eq(StudyDataRequest.ID, Integer.valueOf(id)), StudyDataRequest.class);
		if (studyDataRequest == null) throw new UnknownStudyDefinitionException("Study definition [" + id
				+ "] does not exist");
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("No catalog defined for study definition [" + id + "]");
		setProtocolActive(protocol, true);
	}

	@Override
	public void unloadCatalogOfStudyDefinition(String id) throws UnknownCatalogException,
			UnknownStudyDefinitionException
	{
		StudyDataRequest studyDataRequest = dataService.findOne(StudyDataRequest.ENTITY_NAME,
				new QueryImpl().eq(StudyDataRequest.ID, Integer.valueOf(id)), StudyDataRequest.class);
		if (studyDataRequest == null) throw new UnknownStudyDefinitionException("Study definition [" + id
				+ "] does not exist");
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("No catalog defined for study definition [" + id + "]");
		setProtocolActive(protocol, false);
	}

	@Override
	public boolean isCatalogOfStudyDefinitionLoaded(String id) throws UnknownCatalogException,
			UnknownStudyDefinitionException
	{
		StudyDataRequest studyDataRequest = dataService.findOne(StudyDataRequest.ENTITY_NAME,
				new QueryImpl().eq(StudyDataRequest.ID, Integer.valueOf(id)), StudyDataRequest.class);
		if (studyDataRequest == null) throw new UnknownStudyDefinitionException("Study definition [" + id
				+ "] does not exist");
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("No catalog defined for study definition [" + id + "]");
		return protocol.getActive();
	}

	private void setProtocolActive(Protocol protocol, boolean active)
	{
		protocol.setActive(active);
		dataService.update(Protocol.ENTITY_NAME, protocol);
	}

	@Override
	public void deactivateCatalog(String id) throws UnknownCatalogException
	{
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
		setProtocolActive(protocol, false);
	}

	@Override
	public void activateCatalog(String id) throws UnknownCatalogException
	{
		Protocol protocol = dataService.findOne(Protocol.ENTITY_NAME, new QueryImpl().eq(Protocol.ID, id),
				Protocol.class);
		if (protocol == null) throw new UnknownCatalogException("Catalog [" + id + "] does not exist");
		setProtocolActive(protocol, true);
	}
}
